function phase_calibration
%% Calibrate for DMD phase pattern 

window = 10; % Window size, used in refinement step
[window_v,window_u] = meshgrid(-window:window,-window:window);

num_patterns = 19; % Number of DMD patterns is num_patterns x num_patterns
J = 1:num_patterns; % Indices in X direction
K = 1:num_patterns; % Indices in Y direction

[grid_v,grid_u] = meshgrid(1:1920,1:1080);

% Centroids of DMD block
mtx_pattern_u = zeros(num_patterns);
mtx_pattern_v = zeros(num_patterns);

% Shifts in speckle pattern
mtx_shift_u   = zeros(num_patterns);
mtx_shift_v   = zeros(num_patterns);

hw = waitbar(0,'Running...');

for j = J
    waitbar(j/num_patterns,hw);

    for k = K
        % Read cropped copy of measured image
        img = imread(sprintf('data/measurements/random_warped_%d_%d.jpg',j-1,k-1));
        img = double(img);
        img = img-mean(img(:));
        img = img./norm(img(:));
        
        % Read DMD pattern
        pattern = imread(sprintf('data/patterns/random_pattern_%d_%d.png',j-1,k-1));
        pattern = double(pattern);

        % Propagate wavefront from Fourier plane to image plane
        wave = pattern;
        wave = padarray(wave,[1080,0],'post');
        wave = fftshift(fft2(wave));
        wave = abs(wave(1:1080,:)).^2; % Compute intensity of wavefront
        
        % Crop and normalize
        wave = wave(301:700,301:700);
        wave = wave-mean(wave(:));
        wave = wave./norm(wave(:));

        % Find shift between two images through zero-normalized cross-
        % correlation
        shift = fftshift(abs(ifft2(fft2(img).*conj(fft2(wave)))));
        [val,ind] = max(shift(:));
        [shift_u,shift_v] = ind2sub(size(shift),ind);
        
        % Refine shift (provides subpixel accuracy)
        tmp = shift(shift_u-window:shift_u+window,shift_v-window:shift_v+window);
        du = sum(tmp(:).*window_u(:))./sum(tmp(:));
        dv = sum(tmp(:).*window_v(:))./sum(tmp(:));        
        shift_u = shift_u + du - (size(img,1)./2 + 1);
        shift_v = shift_v + dv - (size(img,2)./2 + 1);
        
        % Compute centroid of pattern
        pattern_u = round(sum(grid_u(:).*pattern(:))./sum(pattern(:)));
        pattern_v = round(sum(grid_v(:).*pattern(:))./sum(pattern(:)));
        
        mtx_pattern_u(j,k) = pattern_u;
        mtx_pattern_v(j,k) = pattern_v;
        mtx_shift_u(j,k) = shift_u;
        mtx_shift_v(j,k) = shift_v;
    end
end

mtx_pattern_u = [mtx_pattern_u(1,:).*0;mtx_pattern_u;mtx_pattern_u(1,:).*0+1080];
mtx_pattern_u = padarray(mtx_pattern_u,[0 1],'replicate');
    
mtx_pattern_v = [mtx_pattern_v(:,1).*0 mtx_pattern_v mtx_pattern_v(:,1).*0+1920];
mtx_pattern_v = padarray(mtx_pattern_v,[1 0],'replicate');
    
mtx_shift_u = padarray(mtx_shift_u,[1 1],'replicate');
mtx_shift_v = padarray(mtx_shift_v,[1 1],'replicate');

% Interpolate shifts across DMD plane
phase_shift_u = interp2(mtx_pattern_v,mtx_pattern_u,mtx_shift_u,grid_v,grid_u,'spline');
phase_shift_v = interp2(mtx_pattern_v,mtx_pattern_u,mtx_shift_v,grid_v,grid_u,'spline');

% Recover phase values that explain shift in speckle pattern
phase_val = 2.*pi.*shift2phase(0.5.*phase_shift_u./1080,phase_shift_v./1920);

% Visualize recovered phase values in HSV colorspace
hsv_phase = hsv2rgb(cat(3,mod(phase_val,2.*pi)./(2.*pi),...
    ones(size(phase_val)),ones(size(phase_val))));
imagesc(hsv_phase);

addpath('./npy-matlab')
writeNPY(phase_val,'calib_dmd_phase.npy');

close(hw);


function usoln = shift2phase(deriv_x,deriv_y)
%% Reconstruct phase from phase gradient

siz = [1080 1920];

mask = ones(size(deriv_x));
mask(1,:) = 0;
ind1 = find(mask);

mask = ones(size(deriv_x));
mask(:,1) = 0;
ind2 = find(mask);

I1 = speye(siz(1));
I2 = speye(siz(2));

e1 = ones(siz(1),1);
e2 = ones(siz(2),1);

R1 = spdiags([-e1 e1],[-1 0],siz(1),siz(1));
R2 = spdiags([-e2 e2],[-1 0],siz(2),siz(2));

B_x = kron(I2,R1);
B_y = kron(R2,I1);

A = [B_x(ind1,:); B_y(ind2,:)];
b = [deriv_x(ind1); deriv_y(ind2)];

A(end+1,1) = 1;
b(end+1)   = 0;

uvec = A\b;  
usoln = reshape(uvec,siz);

