// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

import "@openzeppelin/contracts/token/ERC721/ERC721.sol";
import "@openzeppelin/contracts/token/ERC721/extensions/IERC721Metadata.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/token/ERC721/extensions/ERC721URIStorage.sol";
import "@openzeppelin/contracts/token/ERC721/IERC721Receiver.sol";

contract Rights721 is IERC721Metadata, Ownable, ERC721URIStorage, IERC721Receiver {
  address _owner = msg.sender;
  uint _RightsTypeCount=0;
  uint _ContentID=0;
  uint _uniqueRightID=0;
  uint [] RoyaltyTokenIDs;
  uint RoyaltyTokenCount=0;

  struct NFTs{
    address _originalContractAddress;
    uint _originalID;
    uint _ContentID;
    string _ContentName;
  }

  struct RightsSale{
    address _rightsOwner;
    uint _uniqueRightID;
    uint _ContentID;
    uint _RightID;
    string _RightType;
  }

  struct RoyaltyTokens {
    uint _UniqueRoyaltyRightID;
    address _currentOwner;
    uint256 _percentage;
    uint _ContentTokenID;
  }

  mapping(address => bool) creatorApproval; //holds the address of the content creator and a 0 or 1 for true/false
  mapping(uint => NFTs) public ContentTokens; //holds the content information, user inputs content ID and the original NFT Contract address and ID are returned
  mapping(uint => RightsSale) public SoldRights; //buyers cand check details of their rights; this is also how each rights sale is kept track of
  mapping(uint => string) public RightsType; //this is a list of the rights IDs and which right each ID represents eg 1=RightToTrain
  mapping(address => mapping(uint=>uint)) CoCreatorPercentage;//mapping of colllaborators' addresses to the content they worked on and the percentage of the sale price they should receive(royalties)
  mapping(uint => mapping(uint => uint256)) public PriceOfRight; //this maps the price of each right for each content token
  mapping(address => bool) ApprovedBuyer; //keeps track of approved buyers
  mapping(uint => mapping(uint => bool)) public AvailableRights; //whether a specific right is available for sale for each content token
  mapping(uint => mapping(uint => uint)) public RightsTokensSupply; //total available supply of each right for each content token
  mapping(uint => mapping(uint => uint)) ExercisedRights; //keeps count of the amount of times a certain right has been exercised
  mapping(address => uint256) public UsersCreditBalance; //every user's credit balance
  mapping(uint => uint256) public PriceofRightUsage; //stores the price of exercising of each right
  mapping (uint256 => uint256) public RightForSale; //current right owner indicates that the right is for sale and what the price is
  mapping (uint => RoyaltyTokens) public RoyaltyTokenPay; //Get info about royalty tokens - who holds them and what percentage they get paid
  mapping(uint => address) ContentCreatorMapping; //always holds the address of the content creator

  event ContentCreatorSet(address account, address operator, bool approved);
  event CoCreatorAdded(address account, uint _ContentID, uint _percentage);
  event ContentAdded(address _originaladdress, uint _originalID, uint _internalID, string _ContentName);
  event RightsSold(address _buyer, uint _rightsIntanceID, uint _ContentID, string _RightType);
  event PriceSet(uint _contentid, uint _rightsid, uint _price);
  event BuyerApproved(address _operator, bool _approved);
  event RightTypeAdded(string _rightname, uint _rightid);
  event RightsTransferred(address _from, address _to, uint _rightsIntanceID, uint _ContentID, uint _rightsID);
  event MaxSupplySet(uint _contentid, uint _rightsid, uint _maxSupply);
  event ExercisedRight(uint _contentid, uint _rightsid, uint _amount);
  event AddedCredit(address account, uint256 amount);
  event CreditWithdrawn(address account, uint256 amount);
  event Received(address _operator, address _from,  uint256 _tokenId);
  event RightPostedForSale(uint256 _rightid, uint256 _price);
  event ExercisedRightPriceSet(uint _uniquerightsid, uint256 _usageprice);

  constructor() payable ERC721("Rights721", "ECF") {}
  
  function _onlyContentCreator() private view {
    require(creatorApproval[msg.sender], "5");//"Ownable: caller is not the content creator!");
  }

  modifier onlyContentCreator() {
    _onlyContentCreator();
        _;
    }

  function _onlyRightOwner(uint256 _rightid) private view {
    require(SoldRights[_rightid]._rightsOwner==msg.sender,"6"); //"Ownable: caller is not this right's owner!");
  }
  modifier onlyRightOwner( uint256 _rightid) {
    _onlyRightOwner(_rightid);
        _;
    }

  function _onlyApprovedBuyer() private view {
    require(ApprovedBuyer[msg.sender], "7");//"Ownable: caller is not an approved buyer!");
  }   
  modifier onlyApprovedBuyer() {
    _onlyApprovedBuyer();
        _;
    }
  function isOwner() public view returns (bool) {
    return _msgSender() == _owner;
    } 
    
  function AddCredit() public payable onlyApprovedBuyer{
    UsersCreditBalance[msg.sender]+=msg.value;
    emit AddedCredit(msg.sender, msg.value);
  }

  function WithdrawAllCredit() public payable{
    require(UsersCreditBalance[msg.sender]>= 0, "10");//"The specified account currently has no credit.");
    payable(msg.sender).transfer(UsersCreditBalance[msg.sender]);
    emit CreditWithdrawn(msg.sender, msg.value);
    UsersCreditBalance[msg.sender]=0;
  }
  function onERC721Received(address _operator, address _from, uint256 _tokenId, bytes calldata _data) external override returns(bytes4)  {
    ContentTokens[_ContentID]=NFTs(msg.sender, _tokenId, _ContentID, "placeholder");
    _mint(_from, _uniqueRightID);
    SoldRights[_uniqueRightID]=RightsSale(_from,_uniqueRightID, _ContentID, 0, RightsType[0]);
    _uniqueRightID++;
    emit Received(_operator, _from,  _tokenId);
    emit ContentAdded(_from, _tokenId, _ContentID, "placeholder");
    _ContentID++; 
    _data;
    return 0x150b7a02;
  }

  function SetContentName(uint256 _contentid, string memory _contentName) onlyContentCreator external{
    require(_contentid <=_ContentID, "12");//"The submitted Content ID token does not currently exist!");
    address _from=ContentTokens[_contentid]._originalContractAddress;
    uint256 _tokenid = ContentTokens[_contentid]._originalID;
    ContentTokens[_contentid]=NFTs(_from, _tokenid, _contentid, _contentName);
    emit ContentAdded(_from, _tokenid, _contentid, _contentName);
  }
  function AddRightsType(string memory _rightsType) external onlyContentCreator {
    RightsType[_RightsTypeCount]=_rightsType;
    emit RightTypeAdded(_rightsType, _RightsTypeCount);
    _RightsTypeCount++;
  }
  function SetPrice(uint _contentid, uint _rightsid, uint256 _price) external onlyContentCreator {
    PriceOfRight[_contentid][_rightsid]=_price;
    emit PriceSet(_contentid, _rightsid, _price);
  }

  function SetExercisedRightPrice(uint _uniqueIDofRightToken, uint256 _usageprice) external onlyContentCreator {
    PriceofRightUsage[_uniqueIDofRightToken]=_usageprice;
    emit ExercisedRightPriceSet(_uniqueIDofRightToken, _usageprice);
  }

  function setMaxRightsTokensSupply(uint _contentid, uint _rightsid, uint _maxSupply) external onlyContentCreator {
    RightsTokensSupply[_contentid][_rightsid]=_maxSupply;
        AvailableRights[_contentid][_rightsid]=true;
    emit MaxSupplySet(_contentid, _rightsid, _maxSupply);
  }

  function AddCoCreators(address _cocreator, uint _ContentTokenID, uint _percentage) external onlyContentCreator {
    CoCreatorPercentage[_cocreator][_ContentTokenID]=_percentage;
    _mint(_cocreator, _uniqueRightID);
    RoyaltyTokenCount++;
    SoldRights[_uniqueRightID]=RightsSale(_cocreator,_uniqueRightID, _ContentTokenID, 0,RightsType[1]);
    RoyaltyTokenPay[_uniqueRightID]=RoyaltyTokens(_uniqueRightID, _cocreator, _percentage, _ContentTokenID);
    RoyaltyTokenIDs.push(_uniqueRightID);
    emit CoCreatorAdded( _cocreator,  _ContentTokenID, _percentage);
    _uniqueRightID++;
  }

  function setContentCreator(address _operator, bool _approved) external onlyOwner {
    creatorApproval[_operator] = _approved;
    emit ContentCreatorSet(msg.sender, _operator, _approved);
    ContentCreatorMapping[1]=_operator;
}

  function setApprovedBuyer(address _operator, bool _approved) external onlyContentCreator {
    ApprovedBuyer[_operator] = _approved;
    emit BuyerApproved( _operator, _approved);
  }

  function setTokenURI(uint256 _UniqueRightID, string memory _tokenURI) external onlyContentCreator {
    _setTokenURI(_UniqueRightID, _tokenURI);
  }

  function BuyRightsFromCreator( uint _contentTokenID, uint _rightsID) external  onlyApprovedBuyer {
    require(AvailableRights[_contentTokenID][_rightsID] == true, "13");//"The requested right is not available for sale for this specific content token.");
    require(ApprovedBuyer[msg.sender] == true, "14");//"The address does not belong to a pre-authorised buyer");
    require(_rightsID > 0 , "15");//"The requested right token ID is exclusively reserved for the content creators and cannot be transferred.");
    require(_rightsID <= _RightsTypeCount, "16");//"The requested right token ID does not currently exist.");
    require(_contentTokenID<= _ContentID, "17");//"The requested content token ID does not currently exist.");
    require(RightsTokensSupply[_contentTokenID][_rightsID]>=1, "18");//"The requested right has sold out.");
    require(UsersCreditBalance[msg.sender]>=PriceOfRight[_contentTokenID][_rightsID], "19");//"Insufficient funds, add credit!");
    _mint(msg.sender, _uniqueRightID);
    RightsTokensSupply[_contentTokenID][_rightsID]--;
    SoldRights[_uniqueRightID]=RightsSale(msg.sender,_uniqueRightID, _contentTokenID, _rightsID, RightsType[_rightsID]);
    emit RightsSold(msg.sender,_uniqueRightID, _contentTokenID, RightsType[_rightsID]);
    _ComputeRoyalties(_contentTokenID, PriceOfRight[_contentTokenID][_rightsID]);
    _uniqueRightID++;
    UsersCreditBalance[msg.sender]-=PriceOfRight[_contentTokenID][_rightsID];
  }

  function BuyRightsFromPrivateSeller(uint256 _rightsid) external onlyApprovedBuyer {
    require(RightForSale[_rightsid]>0, "20");//"The requested Right Token is not currently for sale");
    require(UsersCreditBalance[msg.sender]>=RightForSale[_rightsid], "21");//"You need to add more credit in order to buy this right!");
    _safeTransfer(SoldRights[_rightsid]._rightsOwner, msg.sender, _rightsid, "");
    UsersCreditBalance[msg.sender]-=RightForSale[_rightsid];
    UsersCreditBalance[SoldRights[_rightsid]._rightsOwner]+=RightForSale[_rightsid];
    SoldRights[_rightsid]._rightsOwner=msg.sender;
    emit RightsSold(msg.sender, _rightsid, SoldRights[_rightsid]._ContentID, SoldRights[_rightsid]._RightType);
    RightsTokensSupply[SoldRights[_rightsid]._ContentID][SoldRights[_rightsid]._RightID]--;
    _ComputeRoyalties( SoldRights[_rightsid]._ContentID,RightForSale[_rightsid]);
    UsersCreditBalance[ContentCreatorMapping[1]]-=RightForSale[_rightsid];
  }

  function PostRightForSale(uint _rightsid, uint256 _price) external onlyRightOwner(_rightsid) {
    RightForSale[_rightsid]=_price;
    emit RightPostedForSale(_rightsid, _price);
    RightsTokensSupply[SoldRights[_rightsid]._ContentID][SoldRights[_rightsid]._RightID]+=1;
  }

  function _ComputeRoyalties(uint256 _contentid, uint256 _price) internal {
    uint256 total_royalties;
    for (uint i=0; i < RoyaltyTokenCount; i++) {
      if (SoldRights[RoyaltyTokenIDs[i]]._ContentID==_contentid){
        uint256 royalty=(_price * RoyaltyTokenPay[RoyaltyTokenIDs[i]]._percentage)/100;
        UsersCreditBalance[RoyaltyTokenPay[RoyaltyTokenIDs[i]]._currentOwner]+=royalty;
        total_royalties+=royalty;
      }
      
    }
    UsersCreditBalance[ContentCreatorMapping[1]]+=_price-total_royalties;
  }

  function exerciseRights(uint _uniqueRightsID, uint _AmountOfTimes, uint256 _amountPaid) external {
    require(_AmountOfTimes>0, "22"); //you cannot exercise your rights 0 times
    require(SoldRights[_uniqueRightsID]._rightsOwner==msg.sender,"3");//"The account used is not the owner of this right.");
    require(UsersCreditBalance[msg.sender]-_amountPaid>=0, "Add more credit in order to be able to exercise this right at your specified price." );
    UsersCreditBalance[msg.sender]-=_amountPaid;
    ExercisedRights[SoldRights[_uniqueRightsID]._ContentID][SoldRights[_uniqueRightsID]._RightID]+=_AmountOfTimes;
    emit ExercisedRight(SoldRights[_uniqueRightsID]._ContentID, SoldRights[_uniqueRightsID]._RightID, _AmountOfTimes);
    _ComputeRoyalties(SoldRights[_uniqueRightsID]._ContentID, _amountPaid);
  }

  function donateToCreator() external payable {

    UsersCreditBalance[ContentCreatorMapping[1]]+=msg.value;

  }

  function donateForContent(uint256 ContentID) external payable {

    _ComputeRoyalties(ContentID, msg.value);

  }

  function transferOwnership(address) public pure override { revert("disabled"); }
  function safeTransferFrom(address, address, uint256, bytes memory) public pure override(ERC721, IERC721) { revert("disabled"); }
  function safeTransferFrom(address, address, uint256) public pure override(ERC721, IERC721) { revert("disabled"); }
  function transferFrom(address, address, uint256) public pure override(ERC721, IERC721) { revert("disabled"); }
  function setApprovalForAll(address, bool) public pure override(ERC721, IERC721) { revert("disabled"); }
  function approve(address, uint256) public pure override(ERC721, IERC721) { revert("disabled"); }



}
