/*
 * Globally-Optimal Gaussian Mixture Alignment (GOGMA): l2_cost_function
 * L2 cost function for rigid registration. Required for VNL LBFGSB optimiser.
 *
 * Campbell, D., and Petersson, L., "GOGMA: Globally-Optimal Gaussian
 * Mixture Alignment", IEEE Conference on Computer Visision and
 * Pattern Recognition (CVPR), Las Vegas, USA, IEEE, Jun. 2016
 *
 * For the full license, see license.txt in the root directory
 *
 * Author: Dylan Campbell
 * Date: 20160212
 * Revision: 1.1
 */

#include "l2_cost_function.h"

// Constructor: inherits from vnl_cost_function and sets sensible defaults
L2CostFunction::L2CostFunction() : vnl_cost_function() {
  dimension_ = 0;
  sigma_x_ = 0.01;
  sigma_y_ = 0.01;
  gogma_ = NULL;
}

// Evaluates the cost function at theta
// As a side-effect, gradient_ = df/dM is calculated (not df/dtheta)
double L2CostFunction::f(const vnl_vector<double>& theta) {
	gogma_->TransformMuY(theta.data_block());
	double fval = gogma_->GaussTransform(sigma_x_, sigma_y_, gradient_.data_block());
	fval = -fval;
	gradient_ = -gradient_;
	return fval;
}

// Evaluate the gradient at theta
void L2CostFunction::gradf(const vnl_vector<double>& theta, vnl_vector<double>& fgrad) {
	if (dimension_ == 2) {
		fgrad[0] = gradient_.get_column(0).sum();
		fgrad[1] = gradient_.get_column(1).sum();

		vnl_matrix<double> dR;
		dR.set_size(2, 2);
		dR[0][0] = -sin(theta[2]);
		dR[0][1] = -cos(theta[2]);
		dR[1][0] = +cos(theta[2]);
		dR[1][1] = -sin(theta[2]);

		vnl_matrix<double> GmuY;
		GmuY.set_size(dimension_, dimension_);
		GmuY.fill(0.0);
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				for (int k = 0; k < gogma_->num_components_y(); ++k) {
					GmuY[i][j] += gradient_[k][i] * gogma_->mu_y()[k * dimension_ + j];
				}
			}
		}

		fgrad[2] = 0;
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				fgrad[2] += GmuY[i][j] * dR[i][j];
			}
		}
	} else if (dimension_ == 3) {
		fgrad[4] = gradient_.get_column(0).sum();
		fgrad[5] = gradient_.get_column(1).sum();
		fgrad[6] = gradient_.get_column(2).sum();

		vnl_vector<double> q;
		q.set_size(4);
		for (int i = 0; i < 4; ++i) {
			q[i] = theta[i];
		}
		vnl_matrix<double> R;
		R.set_size(3, 3);
		vnl_matrix<double> dR1, dR2, dR3, dR4;
		dR1.set_size(3, 3);
		dR2.set_size(3, 3);
		dR3.set_size(3, 3);
		dR4.set_size(3, 3);
		gogma_->QuaternionToRotation(q.data_block(), R.data_block(), dR1.data_block(), dR2.data_block(), dR3.data_block(), dR4.data_block());

		vnl_matrix<double> GmuY;
		GmuY.set_size(dimension_, dimension_);
		GmuY.fill(0.0);
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				for (int k = 0; k < gogma_->num_components_y(); ++k) {
					GmuY[i][j] += gradient_[k][i] * gogma_->mu_y()[k * dimension_ + j];
				}
			}
		}

		fgrad[0] = 0;
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				fgrad[0] += GmuY[i][j] * dR1[i][j];
			}
		}
		fgrad[1] = 0;
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				fgrad[1] += GmuY[i][j] * dR2[i][j];
			}
		}
		fgrad[2] = 0;
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				fgrad[2] += GmuY[i][j] * dR3[i][j];
			}
		}
		fgrad[3] = 0;
		for (int i = 0; i < dimension_; ++i) {
			for (int j = 0; j < dimension_; ++j) {
				fgrad[3] += GmuY[i][j] * dR4[i][j];
			}
		}
	}
}

// Initialise the class
// Transfers relevant values and sets the number of unknowns
void L2CostFunction::Initialise(GOGMA* gogma) {
	set_gogma(gogma);
	dimension_ = gogma_->dimension();
	sigma_x_ = gogma_->sigma_x();
	sigma_y_ = gogma_->sigma_y();
	gradient_.set_size(gogma_->num_components_y(), dimension_);

	if (dimension_ == 2) {
		set_number_of_unknowns(3); // [x y alpha]
	} else if (dimension_ == 3) {
		set_number_of_unknowns(7); // [qx qy qz qw tx ty tz]
	}
}
