/*
 * Globally-Optimal Gaussian Mixture Alignment (GOGMA): svgm
 * A class for the Support Vector-parametrised Gaussian Mixture (SVGM)
 * data representation. Includes methods for loading point-sets, constructing
 * the SVGM and manipulation.
 *
 * Campbell, D., and Petersson, L., "GOGMA: Globally-Optimal Gaussian
 * Mixture Alignment", IEEE Conference on Computer Visision and
 * Pattern Recognition (CVPR), Las Vegas, USA, IEEE, Jun. 2016
 *
 * For the full license, see license.txt in the root directory
 *
 * Author: Dylan Campbell
 * Date: 20160212
 * Revision: 1.1
 */

#ifndef SVGM_H_
#define SVGM_H_

#include <iostream>
#include <fstream>
#include <time.h>
#include <vector>
#include <algorithm> // random_shuffle, count
#include <vnl/vnl_vector.h>
#include <vnl/vnl_matrix.h>

#include "svm.h"

class SVGM {
public:
	// Constructors
	SVGM();

	// Destructors
	~SVGM();

	// Accessors
	int num_points() const {return num_points_;}
	int num_components() const {return num_components_;}
	int max_num_points() const {return max_num_points_;}
	int min_num_components() const {return min_num_components_;}
	int dimension() const {return dimension_;}
	double sigma() const {return sigma_;}
	double nu() const {return nu_;}
	double sigma_factor() const {return sigma_factor_;}
	struct svm_problem problem() const {return problem_;}
	struct svm_parameter svm_parameters() const {return svm_parameters_;}
	struct svm_model* model() const {return model_;}
	vnl_matrix<double> mu() const {return mu_;}
	vnl_vector<double> phi() const {return phi_;}

	// Mutators
	void set_num_points(int num_points) {num_points_ = num_points;}
	void set_num_components(int num_components) {num_components_ = num_components;}
	void set_max_num_points(int max_num_points) {max_num_points_ = max_num_points;}
	void set_min_num_components(int min_num_components) {min_num_components_ = min_num_components;}
	void set_dimension(int dimension) {dimension_ = dimension;}
	void set_sigma(double sigma) {sigma_ = sigma;}
	void set_nu(double nu) {nu_ = nu;}
	void set_sigma_factor(double sigma_factor) {sigma_factor_ = sigma_factor;}
	void set_problem(struct svm_problem problem) {problem_ = problem;}
	void set_svm_parameters(struct svm_parameter svm_parameters) {svm_parameters_ = svm_parameters;}
	void set_model(struct svm_model* model) {model_ = model;}
	void set_mu(vnl_matrix<double>& mu) {mu_ = mu;}
	void set_phi(vnl_vector<double>& phi) {phi_ = phi;}

	// Public Class Functions
	int LoadPointSet(std::string filename);
	void GetBoundingBox(double centre[3], double widths[3]);
	void TranslatePointSet(double translation[3]);
	void RotatePointSet(double rotation[3][3]);
	void ScalePointSet(double scale);
	void BuildSVGM();

private:
	int num_points_;
	int num_components_;
	int max_num_points_;
	int min_num_components_;
	int dimension_;
	double sigma_, nu_, sigma_factor_;
	struct svm_problem problem_;
	struct svm_parameter svm_parameters_;
	struct svm_model* model_;
	vnl_matrix<double> mu_;
	vnl_vector<double> phi_;

	// Private Class Functions
	void InitialiseSVMParameters();
	int LoadPointSet(std::string filename, struct svm_problem* problem);
	void InitialiseSVM();
	void PrepareSVMProblem(vnl_matrix<double>& pointset, int num_points, int num_dimensions, struct svm_problem* problem);
	bool TrainSVM(const struct svm_problem* problem, struct svm_model** model);
	bool CheckSVMParameters(const struct svm_problem* problem);
	void MapToGMM();
};

#endif /* SVGM_H_ */
