function main
    clc;

    gurobinotfound = startup();
    if gurobinotfound
        return;
    end

    nocallback = testCallback();
    if nocallback
        nocallbackmessage();
    end

    printSep();
    testGORESynthetic(nocallback);

    printSep();
    testGOREAffine(nocallback);
    
    printSep();
    fprintf('Demo concluded.\n\n');
end

function testGORESynthetic(nocallback)
    fprintf('We will now test GORE on pre-generated synthetic data with\n');
    fprintf('T = 10 and c = 15 seconds.\n');
    fprintf('Please see ''./data/genData.m'' on how the data is generated.\n\n');
    
    fprintf('Loading pre-generated data...\n\n');
    load('data/synthetic3D');

    fprintf('Solving this with EXACT alone takes approximately 270 seconds \n')
    fprintf('on a 2.70 GHz machine.\n');
    fprintf('(We will not be running EXACT here.)\n\n');

    fprintf('Running GORE+EXACT...\n');
    [remI,sol,total_runtime] = GORESynthetic(A,b,th,nocallback);
    fprintf('\nThe total runtime of GORE+EXACT is %f seconds.\n',total_runtime);
    gain = 1-(total_runtime/270);
    fprintf('This represents a gain of approximately %.02f%%.\n',100*gain);
    if nocallback
        fprintf('The gain is lower than expected because we are unable to make\n');
        fprintf('use of GUROBI''s callback functionality.\n');
    end
    fprintf('\n');
    
    try
        illustrate3d(A,b,th,remI,sol);
    catch
        fprintf('Unable to draw MATLAB figure for 3d synthetic data.\n\n');
    end
end

function testGOREAffine(nocallback)
    fprintf('We will now test GORE on affine image matching using pre-generated\n');
    fprintf('data with T = 10 and c = 15 seconds.\n');
    fprintf('Please see ''./data/genData.m'' on how the data is generated.\n\n');

    fprintf('Loading pre-generated data...\n\n');
    load('data/graf_affine');

    fprintf('Solving this with EXACT alone takes approximately 280 seconds \n');
    fprintf('on a 2.70 GHz machine.\n');
    fprintf('(We will not be running EXACT here.)\n\n');

    fprintf('Running GORE+EXACT...\n');
    [remI,sol,total_runtime] = GOREAffine(match,th,nocallback);
    fprintf('\nThe total runtime of GORE+EXACT is %f seconds.\n',total_runtime);
    gain = 1-(total_runtime/280);
    fprintf('This represents a gain of approximately %.02f%%.\n',100*gain);
    if nocallback
        fprintf('The gain is lower than expected because we are unable to make\n');
        fprintf('use of GUROBI''s callback functionality.\n');
    end
    fprintf('\n');
    
    try
        illustrateAffine(match,th,remI,sol);
    catch
        fprintf('Unable to draw MATLAB figure for affine image matching.\n\n');
    end
end

function gurobinotfound = startup()
    fprintf('This is the demo program for CVPR 2016 paper submission ID 1299,\n');
    fprintf('intended for reviewing purposes only.\n\n');

    fprintf('This demo program is designed to be executed using MATLAB on\n');
    fprintf('a 64-bit unix machine with GUROBI 6.0.5 installed and working.\n\n');

    fprintf('The program expects to find the GUROBI files in ''./gurobi605''.\n\n');

    gurobinotfound = 0;
    if 7 ~= exist('./gurobi605','file')
        fprintf('Unable to find the GUROBI installation files under ');
        fprintf('''./gurobi605''.\n');
        fprintf('Please copy and paste the GUROBI installation folder into the\n');
        fprintf('current working directory.\n');
        gurobinotfound = 1;
        return;
    end
    
    fprintf('Running setup file for GUROBI''s MATLAB interface...\n');
    if ismac
        cd gurobi605/mac64/matlab;
    elseif isunix
        cd gurobi605/linux64/matlab;
    else
        error('This demo program is designed to run on a unix machine.');
    end
    gurobi_setup;
    cd ../../..;
    fprintf('\n');
end

function nocallbackmessage()
    fprintf('Unable to make use of GUROBI''s callback functionality,\n');
    fprintf('the runtime of GORE will be longer than in the paper.\n\n'); 
end

function printSep()
    fprintf('-------------------------------------------------------\n\n');
end

function nocallback = testCallback()
    nocallback = 0;
    
    if 3 ~= exist('maxcon_milp','file') % attempt to compile if we cant find the file
        fprintf('Unable to find maxcon_milp mex compiled executable.\n');
        fprintf('Attempting to compile maxcon_milp.c using MATLAB MEX.\n\n');
        try
            if ismac
                mex CFLAGS='$CFLAGS -std=c99 -Wall -m64 -g' -I./gurobi605/mac64/include -L./gurobi605/mac64/lib -lgurobi60 maxcon_milp.c
            elseif isunix
                mex CFLAGS='$CFLAGS -std=c99 -Wall -m64 -g' -I./gurobi605/linux64/include -L./gurobi605/linux64/lib -lgurobi60 maxcon_milp.c
            else
                error('Failed to compile maxcon_milp.c');
            end
            fprintf('Compilation successful.\n\n');
        catch
            warning('Failed to compile maxcon_milp.c');
            nocallback = 1;
            return;
        end
        
    end
    
    % test maxcon_milp to make sure that it can actually run properly
    try
        maxcon_milp(int32([1,1]),[5,0],int32(0),10,inf);
    catch
        warning('Encountered error attempting to run maxcon_milp executable mex file.');
        nocallback = 1;
    end
end
