
// dependencies: GUROBI, MATLAB MEX

// This code is written because we want to make use of the callback functionality in GUROBI
// while we run MILP computations on MATLAB.

#include <stdio.h>
#include "mex.h"
#include "matrix.h"
#include "gurobi_c.h"

#define VAR        0
#define A          1
#define SENSE      2
#define B          3
#define TIME_LIMIT 4
#define BOUND_TH   5

#define DISPLAY_OUTPUT   0 // 1 to display and append to output log, 0 to do nothing 
#define OUTPUT_LOG_FILENAME  "results/mixed01LP.log"
#define SAVE_PROBLEM_LOG 0 // 1 to save, 0 to do nothing 
#define PROBLEM_LOG_FILENAME "results/dumpMILPinfo.lp" // THE FILENAME EXTENSION HAS TO END WITH ".lp"

void checkArgs(int nrhs,const mxArray *prhs[]);
int addVariables(GRBmodel *model,int *grb_error,int cont_var_count,int bin_var_count);
int addConstraints(GRBmodel *model,int *grb_error,const mxArray *prhs[]);
int cb(GRBmodel *model,void *cbdata,int where,void *usrdata);
int writeOutput0(GRBmodel *model,int *grb_error,mxArray *plhs[]);
int writeOutput1(GRBmodel *model,int *grb_error,mxArray *plhs[]);
int writeOutput2(GRBmodel *model,int *grb_error,mxArray *plhs[],int cont_var_count);

// Arguments: maxcon_milp(var,A,sense,b,time_limit,bound_th)
// Argument 1: positive int32 array (var)
// 	This concerns the objective function.
// 	array[0] contains count on continuous variables
// 	array[1] contains count on binary variables
// Argument 2: double matrix (A)
// 	The matrix basically contains the LHS of the linear inequalities of the max concensus problem.
// 	Each row of the matrix contains the LHS of a single inequality constraint.
// 	The matrix should have d columns, where d is the sum of all variable counts in Argument 1
// Argument 3: int32 array (sense)
// 	Basically, the direction of the linear inequality.
// 	-1 represents <=
// 	 0 represents ==
// 	 1 represents >=
// 	We expect the sense array to have the same number of entries as the number of rows in Argument 2.
// Argument 4: double array (b)
// 	This is basically the RHS of the linear inequalities.
// 	We also expect this array to have the same number of entries as the number of rows in Argument 2.
// Argument 5: positive double (time_limit)
// 	How long the MILP computation should run for, measured in seconds.
// Argument 6: positive int32 integer (bound_th)
// 	The lowerbound threshold. The MILP computation should terminate and return immediately if its lowerbound crosses this threshold.
// 
// Returns: [ObjBoundC,ObjVal,bestSol]
// ObjBoundC is the lowerbound of the MILP computation.
// ObjVal is the upperbound of the MILP computation.
// bestSol is the current best solution (which corresponds with the upperbound)
void mexFunction(int nlhs,mxArray *plhs[],int nrhs,const mxArray *prhs[]){
	checkArgs(nrhs,prhs);
	/* initialize GUROBI environment */
	GRBenv   *env   = NULL;
	GRBmodel *model = NULL;
	int grb_error = 0;
	if(DISPLAY_OUTPUT) grb_error = GRBloadenv(&env, OUTPUT_LOG_FILENAME);
	else grb_error = GRBloadenv(&env, NULL); // we pass in NULL if we don't want to create/append to a logfile
	if(grb_error){
		mexPrintf("Error trying to create GUROBI environment.\n");
		goto QUIT;
	}
	/* initialize GUROBI model */
	grb_error = GRBnewmodel(env, &model, "0-1_MIXED_LP", 0, NULL, NULL, NULL, NULL, NULL);
	if(grb_error){
		mexPrintf("Error trying to create GUROBI model.\n");
		goto QUIT;
	}
	/* initialize variables to be passed to GUROBI model */
	int *var = (int*)mxGetData(prhs[VAR]);
	const int cont_var_count = var[0];
	const int bin_var_count  = var[1];
	const double time_limit  = *mxGetPr(prhs[TIME_LIMIT]);
	/* add variables to the GUROBI model */
	if( !addVariables(model,&grb_error,cont_var_count,bin_var_count) ){
		goto QUIT; // the function prints its own error message, so we don't need to do any error reporting here 
	}
	/* set objective sense to minimization. GUROBI performs minimization by default, but it wouldn't hurt to make sure. */
	grb_error = GRBsetintattr(model, GRB_INT_ATTR_MODELSENSE, GRB_MINIMIZE);
	if(grb_error){
		mexPrintf("Error trying to set the objective sense to minimization.");
		goto QUIT;
	}
	grb_error = GRBupdatemodel(model); // manually push for an update before adding constraints, or else adding constraints will fail
  	if(grb_error){
  		mexPrintf("Error processing modifications to the GUROBI model - unable to add variables.");
  		goto QUIT;
  	}
  	/* add constraints to the GUROBI model */
  	if( !addConstraints(model,&grb_error,prhs) ){
  		goto QUIT; // the function prints its own error message
  	}
  	if(!DISPLAY_OUTPUT){
  		grb_error = GRBsetintparam(GRBgetenv(model),"OutputFlag",0);
  	}
  	if(SAVE_PROBLEM_LOG){
		GRBupdatemodel(model);
		GRBwrite(model,PROBLEM_LOG_FILENAME);	
  	}
	/* set cut control */
	grb_error = GRBsetintparam(GRBgetenv(model),"Cuts",3); // the GUROBI model gets a copy of the environment it is passed, so we need to call GRBgetenv()
	if(grb_error){
		mexPrintf("Error setting cut control to 3\n");
		goto QUIT;
	}
	grb_error = GRBsetintparam(GRBgetenv(model),"Threads",4);
	if(grb_error){
		mexPrintf("Error setting threadcount to 4\n");
		goto QUIT;
	}
	/* set time limit and other GUROBI parameters */
	grb_error = GRBsetdblparam(GRBgetenv(model),"TimeLimit",time_limit); // the GUROBI model gets a copy of the environment it is passed, so we need to call GRBgetenv() 
	if(grb_error){
		mexPrintf("Error setting time limit of %f seconds.\n",time_limit);
		goto QUIT;
	}
	/* uncomment the following chunk of code to minimize numerical issues.
	   WARNING: may greatly affect runtime and effectiveness of GORE. */
	/*grb_error = GRBsetdblparam(GRBgetenv(model),"IntFeasTol",1e-9);
	if(grb_error){
		mexPrintf("Error setting IntFeasTol limit of %f.\n",1e-9);
		goto QUIT;
	}
	grb_error = GRBsetdblparam(GRBgetenv(model),"FeasibilityTol",1e-9);
	if(grb_error){
		mexPrintf("Error setting FeasibilityTol limit of %f.\n",1e-9);
		goto QUIT;
	}
	grb_error = GRBsetdblparam(GRBgetenv(model),"OptimalityTol",1e-9);
	if(grb_error){
		mexPrintf("Error setting OptimalityTol limit of %f.\n",1e-9);
		goto QUIT;
	}//*/
	/* create callback using the bound_th value, if one is given */
	if(nrhs>=6){
		const int bound_th = *(int*)mxGetData(prhs[BOUND_TH]);
		grb_error = GRBsetcallbackfunc(model,cb,(void*)&bound_th);
		if(grb_error){
			mexPrintf("Error setting the callback function.\n");
			goto QUIT;
		}
	}
	/* RUN OPTIMIZATION */
	grb_error = GRBoptimize(model);
	if(grb_error){
		mexPrintf("Something went wrong while trying to optimize the GUROBI model.\n");
		goto QUIT;
	}
	/* CHECK RESULTS OF OPTIMIZATION */
	int solcount = 0,optimstatus;
	GRBgetintattr(model, GRB_INT_ATTR_SOLCOUNT, &solcount);
	GRBgetintattr(model, GRB_INT_ATTR_STATUS, &optimstatus);
	if(solcount == 0){
		mexPrintf("No solution found, optimization status = %d\n", optimstatus);
	}
	/* retrieve result of computation and write to MATLAB output */
	/* MATLAB OUTPUT 1 */
	if( !writeOutput0(model,&grb_error,plhs) ){
		goto QUIT; // the function prints its own error message
	}
	/* MATLAB OUTPUT 2 */
	if(nlhs >= 2){
		if( !writeOutput1(model,&grb_error,plhs) ){
			goto QUIT; // the function prints its own error message
		}
	}
	/* MATLAB OUTPUT 3 */
	if(nlhs >= 3){
		if( !writeOutput2(model,&grb_error,plhs,cont_var_count) ){
			goto QUIT;
		}
	}

QUIT:
	if(grb_error){
		mexPrintf("GUROBI ERROR %d:%s\n",grb_error,GRBgeterrormsg(env));
	}
	GRBfreemodel(model);
	GRBfreeenv(env);
	if(grb_error){
		mexErrMsgTxt("GUROBI ERROR OCCURRED");
	}
}

// writes the lowerbound of the MILP computation
// returns 1 on success and 0 on failure
int writeOutput0(GRBmodel *model,int *grb_error,mxArray *plhs[]){
	double objBoundC;
	*grb_error = GRBgetdblattr(model,"ObjBoundC",&objBoundC); // in our context this is the lower bound (without rounding)
	if(*grb_error){
		mexPrintf("Error getting ObjBoundC from the GUROBI model.\n");
		return 0;
	}
	plhs[0] = mxCreateDoubleMatrix(1,1,mxREAL);
	double *output = mxGetPr(plhs[0]);
	output[0] = objBoundC;
	return 1;
}
// writes the upperbound of the MILP computation
// returns 1 on success and 0 on failure
int writeOutput1(GRBmodel *model,int *grb_error,mxArray *plhs[]){
	double objVal = mxGetNaN();
	*grb_error = GRBgetdblattr(model,"ObjVal",&objVal); // in our context this is the upper bound. This is also what we return if we want to run LP.
	plhs[1] = mxCreateDoubleMatrix(1,1,mxREAL);
	double *output = mxGetPr(plhs[1]);
	output[0] = objVal;
	if(*grb_error){
		*grb_error = 0;
		//mexPrintf("Error getting ObjVal from the GUROBI model.\n");
	}
	return 1;
}
// writes the current best solution (corresponds to the upper bound)
// returns 1 on success and 0 on failure
int writeOutput2(GRBmodel *model,int *grb_error,mxArray *plhs[],int cont_var_count){
	double sol[cont_var_count];
	*grb_error = GRBgetdblattrarray(model,"X",0,cont_var_count,sol);
	plhs[2] = mxCreateDoubleMatrix(cont_var_count,1,mxREAL);
	double *output = mxGetPr(plhs[2]);
	for(int i = 0; i < cont_var_count; i++){
		if(*grb_error){
			output[i] = mxGetNaN();
		}else{
			output[i] = sol[i];
		}
	}
	if(*grb_error){
		*grb_error = 0;
		//mexPrintf("Error getting variable value of the current best solution.\n");
	}
	return 1;
}

// This is the callback function used for early termination of GUROBI.
// we will be using the usrdata pointer to hold the outlier upper bound.
// whenever this function is invoked, we compare the current best objbst and objbnd to the outlier upper bound and terminate if it is violated.
int cb(GRBmodel *model,void *cbdata,int where,void *usrdata){
	int   bound_th = *((int*)usrdata);
	double objbst; // the current best upper bound from an incumbent solution
	double objbnd; // the current best lower bound
	int    check_bounds = 1, grb_error;
	if(where == GRB_CB_MIP){
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIP_OBJBST, &objbst)) != 0 ) return grb_error;
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIP_OBJBND, &objbnd)) != 0 ) return grb_error;
	}else if(where == GRB_CB_MIPSOL){
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIPSOL_OBJBST, &objbst)) != 0 ) return grb_error;
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIPSOL_OBJBND, &objbnd)) != 0 ) return grb_error;
	}else if(where == GRB_CB_MIPNODE){
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIPNODE_OBJBST, &objbst)) != 0 ) return grb_error;
		if( (grb_error = GRBcbget(cbdata, where, GRB_CB_MIPNODE_OBJBND, &objbnd)) != 0 ) return grb_error;
	}else{
		check_bounds = 0; // no need to check bounds
	}
	if(check_bounds){
		if(bound_th - objbst >= 1.0){ // the optimal solution is guaranteed to be better (less) than bound_th
			GRBterminate(model); // request for termination
		}else if(objbnd > bound_th){ // the optimal solution is guaranteed to be worse (greater) than bound_th
			GRBterminate(model); // request for termination
		}
	}
	return 0;
}

// Adds constraints to the GUROBI model
// returns 1 on success and 0 on failure.
int addConstraints(GRBmodel *model,int *grb_error,const mxArray *prhs[]){
	double *matlab_matrix = mxGetPr(prhs[A]); // the matlab matrix is arranged in the style of an array. Fun.
	const int row_count = mxGetM(prhs[A]);
	const int col_count = mxGetN(prhs[A]);
	int *sense = (int*)mxGetData(prhs[SENSE]);
	double *b  = mxGetPr(prhs[B]);
	for(int row = 0; row < row_count; row++){
		// the LHS of the constraint coefficients are are specified using a list of index-value pairs, one for each non-zero value
		int    numnz = 0;        // the number of nonzero coefficients in LHS of the inequality constraint
		int    index[col_count]; // the index part of the LHS index value pair
		double value[col_count]; // the value part of the LHS index value pair
		char   ineq_symbol;
		double rhs;
		// parse LHS
		for(int col = 0; col < col_count; col++){
			value[numnz] = matlab_matrix[row + col*row_count]; // matlab converts a matrix into a vertical array column by column.
			if(value[numnz] != 0){
				index[numnz] = col;
				numnz++;
			}
		}
		// parse inequality symbol/sense
		ineq_symbol = GRB_EQUAL;
		if     (sense[row] > 0) ineq_symbol = GRB_GREATER_EQUAL;
		else if(sense[row] < 0) ineq_symbol = GRB_LESS_EQUAL;
		// parse RHS
		rhs = b[row];
		// add constraint to GUROBI model
		*grb_error = GRBaddconstr(model,numnz,index,value,ineq_symbol,rhs,NULL);
		if(*grb_error){
			mexPrintf("Error occurred while trying to add constraint in row %d to GUROBI model.\n",row);
			return 0;
		}
	}
	return 1;
}

// Adds variables to the GUROBI model.
// returns 1 on success and 0 on failure
int addVariables(GRBmodel *model,int *grb_error,int cont_var_count,int bin_var_count){
	// ARGUMENTS FOR ADDING CONTINUOUS VARIABLES
	double cont_obj_coef[cont_var_count]; // objective coefficients for continuous variables
	double cont_LB[cont_var_count];       // the lower bound for the continuous variables
	double cont_UB[cont_var_count];       // the upper bound for the continuous variables
	char   cont_vtype[cont_var_count];    // vtype for continuous variables, which should obviously be continuous
	for(int i = 0; i < cont_var_count; i++){
		cont_obj_coef[i] = 0; // continuous variables do not contribute to the objective function
		cont_LB[i] = -1e21;   // any value less than -1e20 is treated as negative infinity. http://www.gurobi.com/documentation/6.0/refman/lb.html#attr:LB
		cont_UB[i] =  1e21;   // any value greater than 1e20 is treated as infinite.        http://www.gurobi.com/documentation/6.0/refman/ub.html#attr:UB
		cont_vtype[i] = GRB_CONTINUOUS;
	}
	// ARGUMENTS FOR ADDING BINARY VARIABLES
	double bin_obj_coef[bin_var_count]; // objective coefficients for binary variables
	//double bin_LB[bin_var_count];       // the lower bound for the binary variables
	//double bin_UB[bin_var_count];       // the upper bound for the binary variables
	char   bin_vtype[bin_var_count];    // vtype for binary variables, which should obviously be binary
	for(int i = 0; i < bin_var_count; i++){
		bin_obj_coef[i] = 1.0;          // binary variables contribute to the objective function by summation 
		bin_vtype[i] = GRB_BINARY;
		//bin_LB[i] = 0;
		//bin_UB[i] = 1;
		//bin_vtype[i] = GRB_CONTINUOUS;
	}
	// ADD CONTINUOUS VARIABLES TO GUROBI MODEL 
	*grb_error = GRBaddvars(model, cont_var_count,
							0, NULL, NULL, NULL,
							cont_obj_coef, cont_LB, cont_UB, cont_vtype,
							//cont_obj_coef, NULL, NULL, cont_vtype,
							NULL);
	if(*grb_error){
		mexPrintf("Error occurred while trying to add continuous variables to GUROBI model.\n");
		return 0;
	}
	// ADD BINARY VARIABLES TO GUROBI MODEL
	// LB and UB can be replaced with NULL to indicate that they should take their default values (0.0 and 1.0 for binary variables)
	*grb_error = GRBaddvars(model, bin_var_count,
							0, NULL, NULL, NULL,
							bin_obj_coef, NULL, NULL, bin_vtype,
							//bin_obj_coef, bin_LB, bin_UB, bin_vtype,
							NULL);
	if(*grb_error){
		mexPrintf("Error occurred while trying to add binary variables to GUROBI model.\n");
		return 0;
	}
	return 1;
}

void checkArgs(int nrhs,const mxArray *prhs[]){
	// inspect arguments
	if(nrhs != 5 && nrhs != 6){
		mexErrMsgTxt("Wrong number of input arguments.\nExpected: maxcon_milp(var,A,sense,b,time_limit,[optional]bound_th)");
	}
	if( !mxIsInt32(prhs[VAR]) ){
		mexErrMsgTxt("Argument 'var' should be an int32 array of length 2!");
	}
	if(mxGetNumberOfElements(prhs[VAR]) != 2){
		mexErrMsgTxt("Argument 'var' should be of length 2!");
	}
	int *var = (int*)mxGetData(prhs[VAR]);
	if(var[0] <= 0 || var[1] <= 0){
		mexErrMsgTxt("Argument 'var' should contain only positive integers greater than 0!");
	}
	if( !mxIsDouble(prhs[A]) ){
		mexErrMsgTxt("Argument 'A' should be of type 'double'!");
	}
	const int var_count = var[0] + var[1];
	if(mxGetN(prhs[A]) != var_count){
		mexErrMsgTxt("Matrix dimensions does not agree! Argument 'A' does not have the right number of columns!");
	}
	if( !mxIsInt32(prhs[SENSE]) ){
		mexErrMsgTxt("Argument 'sense' should be an int32 array!");
	}
	const int row_count = mxGetM(prhs[A]);
	if(mxGetNumberOfElements(prhs[SENSE]) != row_count){
		mexErrMsgTxt("Matrix dimensions does not agree! Argument 'A' and 'sense' should have the same number of entries!");
	}
	if( !mxIsDouble(prhs[B]) ){
		mexErrMsgTxt("Argument 'b' should be of type 'double'!");
	}
	if(mxGetNumberOfElements(prhs[B]) != row_count){
		mexErrMsgTxt("Matrix dimensions does not agree! Argument 'A' and 'b' should have the same number of entries!");
	}
	if( !mxIsDouble(prhs[TIME_LIMIT]) ){
		mexErrMsgTxt("Argument 'time_limit' should be of type 'double'!");
	}
	const double time_limit = *mxGetPr(prhs[TIME_LIMIT]);
	if(time_limit < 0){
		mexErrMsgTxt("Time limit should be greater than 0!");
	}
	if(nrhs >= 6){
		if( !mxIsInt32(prhs[BOUND_TH]) ){
			mexErrMsgTxt("Argument 'bound_th' should be an int32 value!");
		}
		const int bound_th = *(int*)mxGetData(prhs[BOUND_TH]);
		if(bound_th < 0){
			mexErrMsgTxt("Argument 'bound_th' should be greater than or equal to 0!");
		}
	}
}