function lineInfo = calcIntersectAndParallels(epiLines, corrIdxs, imageWidth, imageHeight)
% lineInfo = calcIntersectAndParallels(epiLines, corrIdxs, imageWidth, imageHeight)
%
% Author: Mor Dar
% Last Change: April 4, 2016
%
% This function calculates the intersection points of epipolar lines, the
% parallel lines of each epipolar line from each intersection point and the
% intersections of each of the parallel lines.  It combines all this
% information in te lineInfo structure.

% The time order indices of the epipolar lines.
lineInfo.timeOrderIdxs = corrIdxs;

% This function recieves the epipolar lines, the indices of images with 
% correspondences, and the image size and outputs lines parallel to the 
% epipolar lines, and the intersection points of all the lines.

% The number of epipolar lines.
numLines = size(epiLines,2);

% The number of parallel lines.
numParLines = (numLines^3 - (3 * numLines^2) + (2 * numLines)) / 2;

% Set up the structures for lines and intersections.
epi = struct('id',{},'timeOrderIdx',{},'slope',{},'intercept',{},'color',{},'linewidth',{},'x',{},'y',{});
par = struct('id',{},'epiId',{},'fromIntersectionsOfEpiIds',{},'slope',{},'intercept',{},'color',{},'linewidth',{},'x',{},'y',{});
epiIntersects = struct('intersect',{},'ids',{});
parIntersects = struct('intersect',{},'ids',{});

% Color presets.
colors = [1 0 0; 0 1 0; 0 0 1; 1 0 1; 0 1 1; 1 1 0; 1 1 1; 0 0 0;...
    0.5 0 0; 0 0.5 0; 0 0 0.5; 0.5 0 0.5; 0 0.5 0.5; 0.5 0.5 0; 0.5 0.5 0.5];%hsv(16);

% Start by getting the easy stuff (no calculations needed). Note: Similar
% information for parallel lines is done below during the intersect
% calculations.
for lineNum = 1:numLines
    
    epi(lineNum).id = lineNum;
    epi(lineNum).timeOrderIdx = corrIdxs(lineNum);
    
    epi(lineNum).slope = epiLines{lineNum}.slope;
    epi(lineNum).intercept = epiLines{lineNum}.yintercept;
    epi(lineNum).color = colors(lineNum,:);
    epi(lineNum).linewidth = 3;
end

%% Calculate epipolar line intersections. 

% for intersect calculation
b = [1.0000000000;1.0000000000];

idx = 1;
parIdx = 1;
% for epipolar line a
for line1 = 1:numLines-1  
    
    % for epipolar line b
    for line2 = line1 + 1 : numLines 
        
        % to avoid comparing a line to itsself
        if line1 ~= line2
            
            % Calculate the intersection of the two points.
            a = [epiLines{line1}.epiline(1),epiLines{line1}.epiline(2); ...
                epiLines{line2}.epiline(1),epiLines{line2}.epiline(2)];
            intersect = a\b;
            
            %% Note: we round the intersection points as they cause problems with MATLAB's polybool
            epiIntersects(idx).intersect = [-intersect(1),-intersect(2)];
%             epiIntersects(idx).intersect = [round(-intersect(1)),round(-intersect(2))];
            epiIntersects(idx).ids = [corrIdxs(line1);corrIdxs(line2)];   
            
            % Iterate the index
            idx = idx + 1;
            
            
            %% Calculate the parallel lines (y-intercepts of them).
            % For each line which is neither of these two lines
            for parline = 1:numLines
                
                if parline ~= line1 && parline~= line2
                    % Assign an id, the id of the epipolar line which this
                    % line is parallel to, and the ids of the two epipolar
                    % lines which this epipolar line originates from (the
                    % intersection).
                    par(parIdx).id = parIdx;
                    par(parIdx).epiId = corrIdxs(parline);%parline;
                    par(parIdx).fromIntersectionsOfEpiIds = [corrIdxs(line1);corrIdxs(line2)];%[line1,line2];
                    
                    % The slope is obviously the same as the epipolar line
                    % this line is parallel to, intercept is calculated
                    % using the intersection.
                    par(parIdx).slope = epi(parline).slope;
                    par(parIdx).intercept = (-intersect(2)) - (par(parIdx).slope * (-intersect(1)) );
                    
                    % Color and linewidth match the epipolar line.
                    par(parIdx).color = epi(parline).color;
                    par(parIdx).linewidth = epi(parline).linewidth;
                    
                    % Iterate the parallel index.
                    parIdx = parIdx + 1;                    
                end
            end
        end
    end
end

%% Calculate the parallel line intersections.

idx = 1;
% for each line
for line1 = 1:numParLines-1
    
    % for each other line
    for line2 = line1 + 1 : numParLines
         % to avoid comparing parallel lines or a line to itsself
        if par(line1).epiId ~= par(line2).epiId && line1 ~= line2
            
            % Calculate the intersection point.
            a = [-par(line1).slope,1; -par(line2).slope,1];
            b = [par(line1).intercept;par(line2).intercept];
            intersect = a\b;
            %% Round the intersection points as they cause problems in polybool
            parIntersects(idx).intersect = [intersect(1),intersect(2)];
%             parIntersects(idx).intersect = [round(intersect(1)),round(intersect(2))];
            parIntersects(idx).ids = [line1;line2];   
            idx = idx + 1;      
        end
    end
end




%% For ease of plotting later, we calculate two points on the lines

% We set the enpoints of the lines (xMin, xMax, yMin, yMax) using the
% intersection points of all the lines.  If the maximum or minimum
% intersection happens within the image, we use the image as our limit.

allX = [epiIntersects.intersect parIntersects.intersect];
% Get the minimum and maximum x-values of the intersects.
xMin = min(allX(1:2:end));
xMax = max(allX(1:2:end));

% use the minimum and maximum to extend the endpoints of the lines past the
% intersections.
extentionx = xMax - xMin;
x(1) = xMin - 10*extentionx;
x(2) = xMax + 10*extentionx;

% If intersection points are smaller than the size of the image, use the 
% size of the image as the limits of the lines.
if x(1) > 0
    x(1) = -(10*imageWidth);
end

if x(2) < imageWidth
    x(2) = 10*imageWidth;
end

% For each epipolar line.
for i = 1:numLines
    y(1) = epi(i).slope * x(1) + epi(i).intercept;
    y(2) = epi(i).slope * x(2) + epi(i).intercept;
    epi(i).x = x;
    epi(i).y = y;
end

% For each parallel line
for i = 1:numParLines
    
    y(1) = par(i).slope * x(1) + par(i).intercept;
    y(2) = par(i).slope * x(2) + par(i).intercept;
    par(i).x = x;
    par(i).y = y;
end

% Save all the line information
lineInfo.ylim = [0 imageHeight];
lineInfo.xlim = [0 imageWidth];
lineInfo.epi = epi;
lineInfo.par = par;
lineInfo.parIntersects = parIntersects;
lineInfo.epiIntersects = epiIntersects;
