function regions = calcRegions(lineInfo, imageWidth, imageHeight, idxsCorr, imageIdx)
% function regions = calcRegions(lineInfo, imageWidth, imageHeight, idxsCorr, imageIdx)
%
% Author: Mor Dar
% Last Change: April 4, 2016
%
% This function calculates the regions based on the lines and order.
%
%% Input             Details
%
%  lineInfo         The infomation relating to epipolar and parallel lines
%                   created by calcIntersectAndParallels.
% 
%  imageWidth       The width of the image.
%  imageHeight      The height of the image. 
% 
%  idxsCorr         The indices of the images with correspondence. Note
%                   that as the images are ordered from 1:N, these indices
%                   also correspond to the location of each image in the
%                   time order of image captures. 
%
%  imageIdx         This is the index of the image in which we are
%                   calculating TERs.  As above, it also corresponds to the
%                   location of the image in the temporal order.

%% Output            Details
%  regions           The regions created by the lines.

% The number of invalid TERs is constant.
numInvalidTERs = 8;

% number of lines
numLines = size(lineInfo.epi,2);

%% Initialize some variables

% Instead of image size, use the minimum and maximum intersection values.
minX = Inf; minY = Inf; maxX = -Inf; maxY = -Inf;

temp = [lineInfo.epiIntersects.intersect];
ex = temp(1:2:end); ey = temp(2:2:end);
temp = [lineInfo.parIntersects.intersect];
px = temp(1:2:end); py = temp(2:2:end);

minX = min(min([px ex])-10,0);
minY = min(min([py ey])-10,0);
maxX = max(max([px ex])+10,imageWidth);
maxY = max(max([py ey])+10,imageHeight);

% Create the polygon
imageX = [minX,minX,maxX,maxX,minX];
imageY = [minY,maxY,maxY,minY,minY];

%% Get the regions

% Initialize values.
    regions = [];
    x = cell(1,8);
    y = cell(1,8);
    allImageIds = sort([idxsCorr,imageIdx]);
    
% If there are three epipolar lines
if numLines == 3
    
    % Find the location of the image in the temporal order.
    imageLoc = find(imageIdx == allImageIds);
            
    % Get the line indices for this combination.
    lineIdxs = allImageIds;
    lineIdxs(imageLoc) = [];
    
    % Using the location of the image in the temporal order, look up which
    % regions will be in valid.
    invalidTERs = lookUpInvalidTERs(imageLoc, lineIdxs);
    
    % Calculate each of the eight invalid TERs.
    for invalidTERIdx = 1:numInvalidTERs
        
        % Calculate the invalid TER based on the region type.
        switch invalidTERs{invalidTERIdx}.regionType
            
            case 1 %R1
                [x{invalidTERIdx},y{invalidTERIdx}] = getRegion1(...
                    lineInfo.epiIntersects);
            case 2 %R2
                [x{invalidTERIdx},y{invalidTERIdx}] = getRegion2(...
                    invalidTERs{invalidTERIdx}.lineIdxs, ...
                    lineInfo.epi, lineInfo.epiIntersects);
            case 3 %R3
                [x{invalidTERIdx},y{invalidTERIdx}] = getRegion3(...
                    invalidTERs{invalidTERIdx}.lineIdxs, ...
                    lineInfo.parIntersects, lineInfo.epiIntersects,...
                    lineInfo.par);
            case 4 %R4
                [x{invalidTERIdx},y{invalidTERIdx}] = getRegion4(...
                    invalidTERs{invalidTERIdx}.lineIdxs, lineInfo.par,...
                    lineInfo.parIntersects);
            case 5 %R5
                [x{invalidTERIdx},y{invalidTERIdx}] = getRegion5(...
                    invalidTERs{invalidTERIdx}.lineIdxs, lineInfo);
            otherwise
                
        end
    end
    
    % Combine all the regions into one polygon.
    [rcoordsX,rcoordsY] = PolySetFun([],[],x,y,'union');
        
    % Intersect with the above defined size.
    % Instead of using straight up polybool, use polysetfun to
    % avoid errors due to a matlab bug in the polybool function.
    [invalidTERsX,invalidTERsY] = PolySetFun(imageX,imageY,...
        rcoordsX,rcoordsY,'intersection');
else
    % If there are more than three epipolar lines
    if numLines > 3
        
        % Create all possible groups of 4 (including the point) using the order
        % of all the images without any other reference points.
        groups = nchoosek(allImageIds,4);

        % Remove groupings which do not have the point.
        condition = (groups(:,1) ~= imageIdx & groups(:,2) ~= imageIdx &...
             groups(:,3) ~= imageIdx & groups(:,4) ~= imageIdx);
        groups(condition,:) = [];

        % For each combination of line orders, calculate the region
        % coordiantes.
        for combination = 1:size(groups,1)
            
            % Find the location of the image in the temporal order.
            imageLoc = find(imageIdx == groups(combination,:));
            
            % Get the line indices for this combination.
            lineIdxs = groups(combination,:);
            lineIdxs(imageLoc) = [];
            
            % Using the location of the image in the temporal order, 
            % look up which regions will be invalid.
            invalidTERs = lookUpInvalidTERs(imageLoc, lineIdxs);
            
            % Select the line information for only the relevant lines in
            % this group.
            groupLineInfo = getRelevantLineInfo(lineInfo,...
                groups(combination,:));

            % Calculate each of the eight invalid TERs.
            for invalidTERIdx = 1:numInvalidTERs

                % Calculate the invalid TER based on the region type.
                switch invalidTERs{invalidTERIdx}.regionType

                    case 1 %R1
                        [x{invalidTERIdx},y{invalidTERIdx}] = getRegion1(...
                            groupLineInfo.epiIntersects);
                    case 2 %R2
                        [x{invalidTERIdx},y{invalidTERIdx}] = getRegion2(...
                            invalidTERs{invalidTERIdx}.lineIdxs, ...
                            groupLineInfo.epi, groupLineInfo.epiIntersects);
                    case 3 %R3
                        [x{invalidTERIdx},y{invalidTERIdx}] = getRegion3(...
                            invalidTERs{invalidTERIdx}.lineIdxs, ...
                            groupLineInfo.parIntersects, ...
                            groupLineInfo.epiIntersects,...
                            groupLineInfo.par);
                    case 4 %R4
                        [x{invalidTERIdx},y{invalidTERIdx}] = getRegion4(...
                            invalidTERs{invalidTERIdx}.lineIdxs, ...
                            groupLineInfo.par, groupLineInfo.parIntersects);
                    case 5 %R5
                        [x{invalidTERIdx},y{invalidTERIdx}] = getRegion5(...
                            invalidTERs{invalidTERIdx}.lineIdxs, ...
                            groupLineInfo);
                    otherwise

                end
            end
                
            % Combine all the regions into one polygon.
            [rcoordsX,rcoordsY] = PolySetFun([],[],x,y,'union');
            
            % combine the new invalid regions with the current invalid regions.
            if combination == 1;

                % Instead of using straight up polybool, use polysetfun to
                % avoid errors due to a matlab bug in the polybool function.
                [invalidTERsX,invalidTERsY] = PolySetFun(imageX,imageY,...
                    rcoordsX,rcoordsY,'intersection');
            else
                % Instead of using straight up polybool, use polysetfun to
                % avoid errors due to a matlab bug in the polybool function.
                [a,b] = PolySetFun(imageX,imageY,rcoordsX,rcoordsY,...
                    'intersection');

                % Done this way to avoid errors - slower but more stable 
                % than the straight up polybool union.
                [invalidTERsX,invalidTERsY] = PolySetFun(a,b,...
                    invalidTERsX,invalidTERsY,'union');
            end
        end
                
    end
end
    
% If the whole image is valid (UNLIKELY - but may happen)
if isempty(invalidTERsX)
    regions.invalid(1,:) = [0,0,0];
    regions.invalid(2,:) = [0,0,0];
    regions.valid(1,:) = imageX;
    regions.valid(2,:) = imageY;
else
    % Create outputs of invalid and valid regions.
    regions.invalid(1,:) = invalidTERsX;
    regions.invalid(2,:) = invalidTERsY;

    % Get the valid region by subtracting the valid region from the invalid.
    [tempx,tempy] = PolySetFun(imageX,imageY,...
        invalidTERsX,invalidTERsY,'minus');
    
    % Check if there exists a valid region or if this is a dead end.
    if ~isempty(tempx)
        regions.valid(1,:) = tempx;
        regions.valid(2,:) = tempy;
    else
        regions.valid = [];
    end
end

        
% End of function.
end

function [x,y] = getRegion1(epiIntersects)
% This function returns region number 1 given the intersections of
% epipolar lines.

% Initialize x and y.
x = zeros(1,size(epiIntersects,2)); y = zeros(1,size(epiIntersects,2));

% Region 1 is defined as the three intersections of the epipolar lines.
for idx = 1:size(epiIntersects,2)
    x(idx) = epiIntersects(idx).intersect(1);
    y(idx) = epiIntersects(idx).intersect(2);
end

% Ensure that the region is clockwise.
[x,y] = sortPointsCW(x,y);

% Add the first point to the end, closing the polygon.
x(end+1) = x(1);
y(end+1) = y(1);
end

function [x,y] = getRegion2(lineIdxs, lineHandles, epiIntersects)
% Get the R2 of the two epipolar lines.


% Find the intersection point between the two lines.
ids = [epiIntersects.ids];

intersectIdx = ...
    find((lineIdxs(1) == ids(1,:) & lineIdxs(2) == ids(2,:)) | ...
    (lineIdxs(2) == ids(1,:) & lineIdxs(1) == ids(2,:)));

intersection = epiIntersects(intersectIdx).intersect;

% Calculate the endpoint of each line.

% Start by finding the index of the other point on each line.
otherPointIndex1 = findOtherIntersectPoint(epiIntersects,intersectIdx, lineIdxs(1));
otherPointIndex2 = findOtherIntersectPoint(epiIntersects,intersectIdx, lineIdxs(2));

% Find the endpoint which is closer to the intersection for each line.
endPoint1 = findCloserEndpoint(epiIntersects,intersectIdx,otherPointIndex1,lineHandles,lineIdxs(1));
endPoint2 = findCloserEndpoint(epiIntersects,intersectIdx,otherPointIndex2,lineHandles,lineIdxs(2));

 % Add the points to the valid region.
 x = [intersection(1),endPoint1(1),endPoint2(1)];
 y = [intersection(2),endPoint1(2),endPoint2(2)];

% Ensure that the region is clockwise.
[x,y] = sortPointsCW(x,y);

% Add the first point to the end, closing the polygon.
x(end+1) = x(1);
y(end+1) = y(1);

end

function [x,y] = getRegion3(lineIdxs,parIntersects,epiIntersects,par)
% This function extracts the points which define region type 3.
% 
% lineIdxs: this is a 3 element array of the line ids.  The first index
% should be the epipolar line and the two that follow are the indices of
% the parallel lines (R3 is defined by one epipolar and two parallel lines)

% Get the correct parallel ids as the lineIdxs are based off epiIds.  
parLineIds = [par.id];
epiIds = [par.epiId];
parLineIdxs(1) = parLineIds(epiIds == lineIdxs(1));
parLineIdxs(2) = parLineIds(epiIds == lineIdxs(2));
parLineIdxs(3) = parLineIds(epiIds == lineIdxs(3));

% Find the intersection point between the two parallel lines.
parIds = [parIntersects.ids];

parIntersectIdx = ...
    (parLineIdxs(2) == parIds(1,:) & parLineIdxs(3) == parIds(2,:)) | ...
    (parLineIdxs(3) == parIds(1,:) & parLineIdxs(2) == parIds(2,:));
    
% Get the actual intersection point between the parallel lines.
parIntersection = parIntersects(parIntersectIdx).intersect;


% Find the intersections between the epipolar line and the parallel lines.
epiIds = [epiIntersects.ids]';
intersectIdx = find(epiIds(:,1) == lineIdxs(1) | epiIds(:,2) == lineIdxs(1));

% Extract the intersections (Note: there should only be two as there are
% three lines considered).
intersections(1,:) = epiIntersects(intersectIdx(1)).intersect;
intersections(2,:) = epiIntersects(intersectIdx(2)).intersect;

% build x,y
x = [parIntersection(1), intersections(1,1),intersections(2,1)];
y = [parIntersection(2), intersections(1,2),intersections(2,2)];      

% Ensure that the region is clockwise.
[x,y] = sortPointsCW(x,y);

% Add the first point to the end, closing the polygon.
x(end+1) = x(1);
y(end+1) = y(1);    

% end function.
end


function [x,y] = getRegion4(lineIdxs,parlineHandles, parIntersects)
% This function returns region type 4.

% Get the correct parallel line ids as the lineIdxs are based off epiIds.  
parLineIds = [parlineHandles.id];
epiIds = [parlineHandles.epiId];
parLineIdxs(1) = parLineIds(epiIds == lineIdxs(1));
parLineIdxs(2) = parLineIds(epiIds == lineIdxs(2));

% Getting the intersection between the two lines.
ids = [parIntersects.ids];
intersectionIdx = find((ids(1,:) == parLineIdxs(1) & ids(2,:) == parLineIdxs(2)) |...
    (ids(1,:) == parLineIdxs(2) & ids(2,:) == parLineIdxs(1)));
intersection = parIntersects(intersectionIdx).intersect;

% Calculate the endpoint of each line.

% Start by finding the index of the other point on each line.
otherPointIndex1 = findOtherIntersectPoint(parIntersects, ...
    intersectionIdx, parLineIdxs(1));
otherPointIndex2 = findOtherIntersectPoint(parIntersects, ...
    intersectionIdx, parLineIdxs(2));

% Find the endpoint which is closer to the intersection for each line.
endPoint1 = findCloserEndpoint(parIntersects,intersectionIdx, ...
    otherPointIndex1,parlineHandles,parLineIdxs(1));
endPoint2 = findCloserEndpoint(parIntersects,intersectionIdx, ...
    otherPointIndex2,parlineHandles,parLineIdxs(2));

% Add the points to the valid region.
x = [intersection(1),endPoint1(1),endPoint2(1)];
y = [intersection(2),endPoint1(2),endPoint2(2)];

% Ensure that the region is clockwise.
[x,y] = sortPointsCW(x,y);

% Add the first point to the end, closing the polygon.
x(end+1) = x(1);
y(end+1) = y(1);         

% end function.
end


function [x,y] = getRegion5(lineIdxs, lineInfo)
% This function returns the polygon of region 5.
% Note: lineIdxs(1) = the epipolar line of this region
% lineIdxs(2) = the parallel line which goes to infinity
% lineIdxs(3) = the other parallel line.

% Get the correct parallel line ids as the lineIdxs are based off epiIds.  
parLineIds = [lineInfo.par.id];
epiIds = [lineInfo.par.epiId];
parLineIdxs(1) = parLineIds(epiIds == lineIdxs(1));
parLineIdxs(2) = parLineIds(epiIds == lineIdxs(2));
parLineIdxs(3) = parLineIds(epiIds == lineIdxs(3));

% Get the intersection between the two parallel lines.
parIds = [lineInfo.parIntersects.ids];
parIntersectionIdx = find((parIds(1,:) == parLineIdxs(2) & parIds(2,:) == parLineIdxs(3)) |...
    (parIds(1,:) == parLineIdxs(3) & parIds(2,:) == parLineIdxs(2)));
parIntersection = lineInfo.parIntersects(parIntersectionIdx).intersect;
  

% Find the endpoint of the line parallel to the epipolar line
% (lineIdxs(2)).
otherPointIndex1 = findOtherIntersectPoint(lineInfo.parIntersects,...
    parIntersectionIdx, parLineIdxs(2));
parEndpoint = findCloserEndpoint(lineInfo.parIntersects,...
    parIntersectionIdx,otherPointIndex1,lineInfo.par,parLineIdxs(2));
    
% Get the id of the line which does not border this region (Note:
% although this line does not border the region its intersection with
% lineIdxs(1) does define a corner of the region which is why it is
% important).
epiIds = [lineInfo.epi.timeOrderIdx];
otherLineId = setdiff(epiIds,lineIdxs);
    
% Find the intersection between this line and the epipolar line.
epiIds = [lineInfo.epiIntersects.ids];
epiIntersectionIdx = find(...
    (epiIds(1,:) == otherLineId & epiIds(2,:) == lineIdxs(1)) | ...
    (epiIds(2,:) == otherLineId & epiIds(1,:) == lineIdxs(1)));
epiIntersection = lineInfo.epiIntersects(epiIntersectionIdx).intersect;

% Find the endpoint of the epipolar line.
otherPointIndex2 = findOtherIntersectPoint(...
    lineInfo.epiIntersects,epiIntersectionIdx, lineIdxs(1));
epiEndpoint = findCloserEndpoint(...
    lineInfo.epiIntersects,epiIntersectionIdx,otherPointIndex2,...
    lineInfo.epi,lineIdxs(1));    
    
% Get the polygon points (x,y)
x = [epiEndpoint(1),epiIntersection(1),parIntersection(1),parEndpoint(1)];
y = [epiEndpoint(2),epiIntersection(2),parIntersection(2),parEndpoint(2)];

% Ensure that the region is clockwise.
[x,y] = sortPointsCW(x,y);

% Add the first point to the end, closing the polygon.
x(end+1) = x(1);
y(end+1) = y(1);    

% end function.
end

% This function finds the index of the other intersection point along the
% line with the given line label.
function otherPointIndex = findOtherIntersectPoint(intersects,...
    thisPointIndex, lineLabel)

for idx = 1:size(intersects,2)
    if idx ~= thisPointIndex && max(intersects(idx).ids == lineLabel) == 1
        otherPointIndex = idx;
    end
end
end


% This function finds the closer endpoint on the line.
function endPoint = findCloserEndpoint(epiIntersects,idx,otherPointIndex,...
    lineHandles,lineLabel)

% Start by getting the two endpoints of the line.
if isfield(lineHandles,'timeOrderIdx') % If this is epipolar lines use the time index
    l = lineHandles([lineHandles.timeOrderIdx] == lineLabel);
else % If this is parallel lines use the ids.
    l = lineHandles([lineHandles.id] == lineLabel);
end

linePoints = [l.x;l.y];

% Calculate Euclidean distances.
intersectionDist = norm(epiIntersects(idx).intersect' - ...
    linePoints(:,1));
otherIntersectDist = norm(epiIntersects(otherPointIndex).intersect' - ...
    linePoints(:,1));

% if the intersection is closer than the other point
if intersectionDist < otherIntersectDist 
    endPoint = [linePoints(1,1),linePoints(2,1)];
else
    endPoint = [linePoints(1,2),linePoints(2,2)];
end

% end function.
end

function groupLineInfo = getRelevantLineInfo(lineInfo, indices)
% This function returns only the information from lineInfo which
% corresponds with the indices of the set we are currently considering.

groupLineInfo = lineInfo;

% Find the ids of the epipolar lines which are not represeneted in the
% indices.  
timeIdToRemove = setdiff([lineInfo.epi.timeOrderIdx],indices);

% Remove the epipolar line(s)
groupLineInfo.epi(ismember([lineInfo.epi.timeOrderIdx],timeIdToRemove)) = [];

% Remove the parallel line(s)
temp = [groupLineInfo.par.fromIntersectionsOfEpiIds];
parOffEpiIds(1,:) = temp(1:2:end);
parOffEpiIds(2,:) = temp(2:2:end);

parLinesToRemoveIdxs = ismember([lineInfo.par.epiId],timeIdToRemove) | ...
    ismember(parOffEpiIds(1,:),timeIdToRemove) |...
    ismember(parOffEpiIds(2,:),timeIdToRemove);

% Get the ids in order to remove the correct par intersections.
parIds = [groupLineInfo.par.id];
parIdsToRemove = parIds(parLinesToRemoveIdxs);

groupLineInfo.par(parLinesToRemoveIdxs) = [];

% Remove the epipolar line intersections
epiIntersectIds = [groupLineInfo.epiIntersects.ids];
groupLineInfo.epiIntersects(ismember(epiIntersectIds(1,:),timeIdToRemove) |...
    ismember(epiIntersectIds(2,:),timeIdToRemove)) = [];

% Finally remove the parallel line intersections
parIntersectIds = [groupLineInfo.parIntersects.ids];
groupLineInfo.parIntersects(ismember(parIntersectIds(1,:),parIdsToRemove) |...
    ismember(parIntersectIds(2,:),parIdsToRemove)) = [];

% end function.
end
