function [TERs,p] = runTERs(images,F,varargin)
% TERs = runTERs(images,F,varargin)
%
% Author: Mor Dar
% Last Change: April 3, 2016
%
% This function calcuates TERs in all images for which correspondence is
% unknown.
%
% Input     Details
%
% images    This function recieves the image data as a 1xN cell array. 
%           
%           Each cell should have the following: 
%               images{idx}.image       The actual image (ie. output of imread).
%               images{idx}.imageName   The name of the image (used only
%                                       for displaying purposes at the end).
%               images{idx}.f           (optional)The feature points (2xM 
%                                       matrix where the first row is 
%                                       x-coordinates and the second is y). 
%                                       This is used when selecting initial 
%                                       correspondences and displaying 
%                                       results at the end.
%
% F             This is a NxN cell matrix in which each cell is the
%               fundamental matrix between the image whose index is the row
%               and the image whose index is the column.  
%
% Optional Input:              
%
% p             This is a 2xN matrix consisting of the (x,y) coordinates of
%               corresponding points in a small subset of the the images. 
%               Images in which the correspondence is unknown should have 
%               their column set to [0;0].  If this is not supplied, the
%               program will open a window for manual selection of initial
%               correspondences.
%
% Output        Details
%
% TERs          This is a 1xN cell array in which the calculated TERs are
%               located in the cell indices which correspond to the images 
%               in which we search for correspondence. 
%               Each cell contains a struct with 2 fields:
%               lineInfo        This is a struct which includes all the
%                               epipolar line information.
%               regions         This is a struct which contains two
%                               matrices: valid and invalid.  Each contains
%                               the coordinates of TERs in clockwise order.

% If only two arguments were passed in, use the manual selection GUI to
% manually select an initial set of correspondences.  
if nargin == 2
    p = selectInitCorr(images);
else
    p = varargin{1};
end

% Get the number of correspondences given.
numInitCorr = numel(find(p(1,:)>0));

% If there are fewer than the minimum number of correspondences given,
% return an error.
if numInitCorr < 3
    errordlg('A minimum of 3 initial correspondences must be given for TER calculation.');
    return;
end

% Calculate the TERs in the images without correspondences.
TERs = calculateTERs(images, p, F);

% Display the results.
displayResults(images,p,TERs)