function TERs = calculateTERs(images, p, F)
% handles = calculateTERs(images,F)
%
% Author: Mor Dar
% Last Change: April 4, 2016
%
% This function calcuates TERs in all images for which correspondence is
% unknown.
%
% Input         Details
%
% images        This function recieves the images as a 1xN cell array. The 
%               actual images should be stored in images{idx}.image.
%
% p             This is a 2xN matrix consisting of the (x,y) coordinates of
%               corresponding points in each of the images. Images in which
%               the correspondence is unknown should have their column set
%               to [0;0];
%
% F             This is a NxN cell matrix in which each cell is the
%               fundamental matrix between the image whose index is the row
%               and the image whose index is the column.  
%
% Output        Details
%
% TERs          This is a 1xN cell array in which the calculated TERs are
%               located in the cell indices which correspond to the images 
%               in which we search for correspondence. 
%               Each cell contains a struct with 2 fields:
%               lineInfo        This is a structure which includes all the
%                               epipolar line information.
%               regions         This is a struct which contains two
%                               matrices: valid and invalid.  Each contains
%                               the coordinates of TERs in clockwise order.

% Initialize the output.
TERs = cell(size(images));

% Identify which images do and do not have correspondences.
idxsCorr = find(p(1,:) ~= 0 & p(2,:) ~= 0);
idxsNoCorr = find(p(1,:) == 0 & p(2,:) == 0);

% Convert p to homogeneous coordinates.
p(3,:) = ones(size(p(1,:)));

% For each image in which correspondence is unknown, calculate TERs.
for noCorrIdx = 1:size(idxsNoCorr,2)
    
    % Initialize the line structures.
    TERs{idxsNoCorr(noCorrIdx)}.lineInfo = []; 
    TERs{idxsNoCorr(noCorrIdx)}.regions = [];
    epiLines = cell(size(idxsCorr)); 
    
    % Calculate the epipolar lines from each of the images with
    % correspondence.
    for corrIdx = 1:size(idxsCorr,2)    
        
        epiLines{corrIdx} = calcEpiLine(...
            p(:, idxsCorr(corrIdx)),...
            F{idxsCorr(corrIdx),idxsNoCorr(noCorrIdx)});
        
    end
    
    % Using the epipolar lines, calculate intersections of epipolar lines,
    % parallel lines passing through these intersections and the
    % intersections of parallel lines.  In other words, get all the points
    % which determine region boundaries.
    TERs{idxsNoCorr(noCorrIdx)}.lineInfo = calcIntersectAndParallels(epiLines,...
        idxsCorr,size(images{noCorrIdx}.image,2),size(images{noCorrIdx},1));
    
    % Delete the previous cells to prepare for next iteration.
    epiLines = [];   
    
    %% Calculate the regions
    TERs{idxsNoCorr(noCorrIdx)}.regions = calcRegions(TERs{idxsNoCorr(noCorrIdx)}.lineInfo, ...
        size(images{noCorrIdx}.image,2), size(images{noCorrIdx}.image,1),...
        idxsCorr, idxsNoCorr(noCorrIdx));

end
