function sols = minimal_polynomial_solver(u1,u2,p1,p2,dl)
% Input:
% * u1, u2, p1, p2, each on the form 3x3, such that
%	U_1k = p1(:,k) + t*u1(:,k) and 
%	U_2k = p2(:,k) + t*u2(:,k)
%	define the rays from observations in camera 1 and 2, respectively.
%
% *	dl: a 3-vector with differences in depth, i.e.
%	dl(k) = norm(R*U_k+t-p2(:,k)) - norm(U_k - p1(:,k)),
%	where R and t are unknown.
% 
% Output: 
% * sols: a 3xN matrix where each column are the absolute distances from
%	the unknown scene points to the points p1, i.e.
%	z1(k,i) = norm(U_k - p1(:,k)),
%	where k is the point index and i is the index of the solution set.
%
% Note 1: u1, u2, p1, p2 are defined in the local coordinate systems of the
%	cameras.
% Note 2: for a pinhole camera model, set 
%	p1 = zeros(3,3) and
%	p2 = zeros(3,3)
% Note 3: Since u1 and u2 always have unit length, and p1 and p2 can have
% any size depending on the coordinate system, some weighting might be
% necessary.

% If empty p1 and p2: assume pinhole camera model
if(isempty(p1))
	p1 = zeros(3,3);
end
if(isempty(p2))
	p2 = zeros(3,3);
end

% Make sure vectors have length 1
u1 = u1./repmat(sqrt(sum(u1.^2,1)),[size(u1,1),1]);
u2 = u2./repmat(sqrt(sum(u2.^2,1)),[size(u2,1),1]);
p2 = p2 + u2*diag(dl);

u1u2 = u1(:,1)'*u1(:,2);
u1u3 = u1(:,1)'*u1(:,3);
u2u3 = u1(:,2)'*u1(:,3);
v1v2 = u2(:,1)'*u2(:,2);
v1v3 = u2(:,1)'*u2(:,3);
v2v3 = u2(:,2)'*u2(:,3);
u1p1 = u1(:,1)'*p1(:,1);
u1p2 = u1(:,1)'*p1(:,2);
u1p3 = u1(:,1)'*p1(:,3);
u2p1 = u1(:,2)'*p1(:,1);
u2p2 = u1(:,2)'*p1(:,2);
u2p3 = u1(:,2)'*p1(:,3);
u3p1 = u1(:,3)'*p1(:,1);
u3p2 = u1(:,3)'*p1(:,2);
u3p3 = u1(:,3)'*p1(:,3);
v1q1 = u2(:,1)'*p2(:,1);
v1q2 = u2(:,1)'*p2(:,2);
v1q3 = u2(:,1)'*p2(:,3);
v2q1 = u2(:,2)'*p2(:,1);
v2q2 = u2(:,2)'*p2(:,2);
v2q3 = u2(:,2)'*p2(:,3);
v3q1 = u2(:,3)'*p2(:,1);
v3q2 = u2(:,3)'*p2(:,2);
v3q3 = u2(:,3)'*p2(:,3);

% A11 = -2*(u1(:,1)'*u1(:,2)-u2(:,2)'*u2(:,1));
A11 = -2*(u1u2-v1v2);
A14 = -2*(u1p2-u1p1-v1q2+v1q1);
A15 = 2*(u2p2-u2p1-v2q2+v2q1);
A17 = (p1(:,2)-p1(:,1))'*(p1(:,2)-p1(:,1))-(p2(:,2)-p2(:,1))'*(p2(:,2)-p2(:,1));
A21 = -(u1u2-v1v2);
A22 = -(u1u3-v1v3);
A23 = (u2u3-v2v3);
A24 = -(u1p3-u1p1+u1p2-u1p1-v1q3+v1q1-v1q2+v1q1);
A25 = u2p3-u2p1-v2q3+v2q1;
A26 = u3p2-u3p1-v3q2+v3q1;
A27 = (p1(:,3)-p1(:,1))'*(p1(:,2)-p1(:,1))-(p2(:,3)-p2(:,1))'*(p2(:,2)-p2(:,1));
A32 = -2*(u1u3-v1v3);
A34 = -2*(u1p3-u1p1-v1q3+v1q1);
A36 = 2*(u3p3-u3p1-v3q3+v3q1);
A37 = (p1(:,3)-p1(:,1))'*(p1(:,3)-p1(:,1))-(p2(:,3)-p2(:,1))'*(p2(:,3)-p2(:,1));

C0 = zeros(3,7);
C0(1,1) = A11;
C0(1,4) = A14;
C0(1,5) = A15;
C0(1,7) = A17;
C0(2,1) = A21;
C0(2,2) = A22;
C0(2,3) = A23;
C0(2,4) = A24;
C0(2,5) = A25;
C0(2,6) = A26;
C0(2,7) = A27;
C0(3,2) = A32;
C0(3,4) = A34;
C0(3,6) = A36;
C0(3,7) = A37;

% Solve
C = C0;
C2 = C(:,1:3)\C;

monv = [     1     1     1     0     1     0     0     0
			 1     1     0     1     0     1     0     0
			 1     0     1     1     0     0     1     0];
Cs = [zeros(3,1),C2];

Cs2 = zeros(size(Cs));
Cs2(:,1) = 1;
Cs2(1,3) = C2(1,4);
Cs2(1,4) = C2(1,5);
Cs2(1,7) = C2(1,7);
Cs2(2,2) = C2(2,4);
Cs2(2,4) = C2(2,6);
Cs2(2,6) = C2(2,7);
Cs2(3,2) = C2(3,5);
Cs2(3,3) = C2(3,6);
Cs2(3,5) = C2(3,7);
C = [Cs;Cs2];

% 
actmon = [1;0;0];
B = [0 0 0;1 0 0;0 1 0];

R = B + repmat(actmon,1,size(B,2));


% find indices for B and R monomials in the full monomial list
Bi = zeros(1,size(B,2));
for kk = 1:size(B,2);
    Bi(kk) = find(all(repmat(B(:,kk),1,size(monv,2))==monv));
end
Ri = zeros(1,size(R,2));
for kk = 1:size(R,2);
    Ri(kk) = find(all(repmat(R(:,kk),1,size(monv,2))==monv));
end

%
B2 = Bi;
R2 = setdiff(Ri,Bi);
E2 = setdiff(1:size(monv,2),union(B2,R2));

% Reorder the monomials into the excessive, reducible and basis
% V2 = [E2 R2 B2];
C2 = C(:, [E2, R2, B2]);

[qq,~,~] = qr(C2(:, 1 : length(E2)));
kk = 2;

C4R = (qq(:,(kk+1):end)')*C(:,R2);
C4B = (qq(:,(kk+1):end)')*C(:,B2);

%
bn = length(B2);

RR = C4R\C4B;
RRR = [-RR' eye(bn)];

monR = monv(:,R2);
monB = monv(:,B2);
monRB = [monR monB];

%B = m2basis;
Bx = B + repmat(actmon,1,size(B,2));

Bxi = zeros(1,size(Bx,2));
for kk = 1:size(Bx,2);
    Bxi(kk) = find(all(repmat(Bx(:,kk),1,size(monRB,2))==monRB));
end

AM = RRR(:,Bxi)';
[V,D]=eig(AM);

% Extract solutions
V = V./repmat(V(3,:),3,1);
sols = [diag(D)';V(1:2,:)];
sols = sols.*repmat(sign(sols(end,:)),[size(sols,1),1]);

% Remove negative solutions
isBad = any(sols<0,1);
sols(:,isBad) = [];

% Validate solutions
res_all = zeros(size(sols));
for k = 1:size(sols,2)
	sk = sols(:,k);
	vx = [sk(1)*sk(2);sk(1)*sk(3);sk(2)*sk(3);sk;1];
	res_all(:,k) = C0*vx;
end

% Remove bad solutions
isBad = sqrt(sum(res_all.^2,1))>1e-10;
sols(:,isBad) = [];
