import torch
import torch.nn as nn
from NewQuantizationBlock import QuantizationBlock
"""code for Anonymous CVPR submission: 
ResNet models quantized via QuantizationBlock """

"""ResNet model was borrowed from https://github.com/pytorch/vision/blob/master/torchvision/models"""



def forward_module_list(mod_list, input, features, collect_statistics=False):
    """Forward module lists nested up to twice"""
    out = input
    for i, mod_t in enumerate(mod_list):
        if isinstance(mod_t, nn.Sequential):
            for j, layer in enumerate(mod_t):
                if isinstance(layer, QuantizationBlock):
                    features = layer.predict_qp(out,features)
                    out = layer(out, collect_statistics=collect_statistics)
                elif isinstance(layer, Bottleneck) or isinstance(layer, BasicBlock):
                    out, features = layer(out, features, collect_statistics=collect_statistics)
                else:
                    out = layer(out)
        else:
            if isinstance(mod_t, QuantizationBlock):
                features = mod_t.predict_qp(out, features)
                out = mod_t(out, collect_statistics=collect_statistics)
            elif isinstance(mod_t, Bottleneck) or isinstance(mod_t, BasicBlock):
                out, features = mod_t(out, features, collect_statistics=collect_statistics)
            else:
                out = mod_t(out)
    return out, features

def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1
    def __init__(self, inplanes, planes, QB_CONFIG, name, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.q_block_1 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_first_layer_basic_block')
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        if downsample is None:
            self.q_block_2 = None
        else:
            self.q_block_2 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_downsample_basic_block')

        self.q_block_3 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_output_basic_block')
        self.stride = stride

    def forward(self, x, features, collect_statistics=False):
        residual = x
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)
        features = self.q_block_1.predict_qp(out, features)
        out = self.q_block_1(out, collect_statistics=collect_statistics)
        out = self.conv2(out)
        out = self.bn2(out)
        if self.downsample is not None:
            features =self.q_block_2.predict_qp(out,features)
            out = self.q_block_2(out, collect_statistics=collect_statistics)
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)
        features = self.q_block_3.predict_qp(x,features)
        out = self.q_block_3(out, collect_statistics=collect_statistics)

        return out, features


class Bottleneck(nn.Module):

    expansion = 4

    def __init__(self, inplanes, planes, QB_CONFIG, name, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.q_block_1 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_first_layer_bottleneck_block')

        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.q_block_2 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_second_layer_bottleneck_block')

        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.q_block_4 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id=name + '_output_bottleneck_block')

        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        if self.downsample is None:
            self.q_block_3 = None
        else:
            self.q_block_3 = QuantizationBlock(QB_CONFIG, is_quantized=True,
                                               qblock_id=name + '_downsample_layer_bottleneck_block')

        self.stride = stride

    def forward(self,x, features, collect_statistics=False):
        identity = x
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)
        features = self.q_block_1.predict_qp(out, features)
        out = self.q_block_1(out, collect_statistics=collect_statistics)
        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)
        features = self.q_block_2.predict_qp(out, features)
        out = self.q_block_2(out, collect_statistics=collect_statistics)
        out = self.conv3(out)
        out = self.bn3(out)
        if self.downsample is not None:
            features = self.q_block_3.predict_qp(out, features)
            out = self.q_block_3(out, collect_statistics=collect_statistics)
            identity = self.downsample(x)
        out += identity
        out = self.relu(out)
        features = self.q_block_4.predict_qp(out, features)
        out = self.q_block_4(out, collect_statistics=collect_statistics)
        return out, features


class ResNet(nn.Module):

    def __init__(self, QB_CONFIG, block, layers, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 norm_layer=None):
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.q_block_start_1 = QuantizationBlock(QB_CONFIG, is_quantized=False, qblock_id='input_qblock')

        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.conv1.quantize_weights = False
        self.q_block_start_2 = QuantizationBlock(QB_CONFIG, is_quantized=True, qblock_id='prelayer_qblock_1')
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        name = 'layer1'
        self.layer1 = self._make_layer(block, 64, layers[0],QB_CONFIG, name)
        name = 'layer2'
        self.layer2 = self._make_layer(block, 128, layers[1], QB_CONFIG, name,stride=2,
                                       dilate=replace_stride_with_dilation[0])
        name = 'layer3'
        self.layer3 = self._make_layer(block, 256, layers[2], QB_CONFIG, name,stride=2,
                                       dilate=replace_stride_with_dilation[1])
        name = 'layer4'
        self.layer4 = self._make_layer(block, 512, layers[3], QB_CONFIG, name,stride=2,
                                       dilate=replace_stride_with_dilation[2])

        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)
        self.fc.quantize_weights = False
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, QB_CONFIG, name, stride=1, dilate=False):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, QB_CONFIG, name + '_bl0', stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            name_temp = name + '_bl' + str(i)
            layers.append(block(self.inplanes, planes, QB_CONFIG, name_temp, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)


    def _forward_impl(self, x_init, collect_statistics=False):
        # See note [TorchScript super()]
        x = x_init
        self.features = (torch.tensor([]), torch.tensor([]))
        self.features = self.q_block_start_1.predict_qp(x, self.features)
        x = self.q_block_start_1(x, collect_statistics=collect_statistics)
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)
        self.features = self.q_block_start_2.predict_qp(x, self.features)
        x = self.q_block_start_2(x, collect_statistics=collect_statistics)

        x, self.features = forward_module_list(self.layer1, x, features=self.features,
                                               collect_statistics=collect_statistics)
        x, self.features = forward_module_list(self.layer2, x, features=self.features,
                                               collect_statistics=collect_statistics)
        x, self.features = forward_module_list(self.layer3, x, features=self.features,
                                               collect_statistics=collect_statistics)
        x, self.features = forward_module_list(self.layer4, x, features=self.features,
                                               collect_statistics=collect_statistics)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)
        return x

    def forward(self, x, collect_statistics=False):
        return self._forward_impl(x, collect_statistics)

    def proccess_model_statistics(self):
        for name, m in self.named_modules():
            if isinstance(m, QuantizationBlock):
                if m.getattr_statistic_container() is not None and len(m.getattr_statistic_container()['MEAN']) != 0:
                    m.process_statistics_container()

    def dump_all_dense_fractions(self):
        for name, m in self.named_modules():
            if isinstance(m, QuantizationBlock):
                m.dump_dense_fraction()


def _resnet(QB_config, block, layers, **kwargs):
    model = ResNet(QB_config, block, layers, **kwargs)
    # if pretrained:
    #     state_dict = load_state_dict_from_url(model_urls[arch],
    #                                           progress=progress)
    #     model.load_state_dict(state_dict)
    return model

def resnet18(QB_config, **kwargs):

    return _resnet(QB_config, BasicBlock, [2, 2, 2, 2],
                   **kwargs)

def resnet34(QB_config, **kwargs):

    return _resnet(QB_config,  BasicBlock, [3, 4, 6, 3],
                   **kwargs)