#pragma once

#include <cosy/affine.h>
#include <cosy/proj.h>
#include <xtensor/xtensor.hpp>
#include <xti/opencv.h>
#include <xti/util.h>
#include <xtensor-blas/xlinalg.hpp>
#include <tiledwebmaps/layout.h>
#include "../exception.h"
#include "camera.h"
#include "lidar.h"
#include <cereal/access.hpp>
#include <xti/cereal.h>
#include <cereal/types/memory.hpp>

namespace georegdata::ground {

class Frame;
class AlignedFrameId;
class AlignedFrame;

class Imu
{
public:
  Imu()
    : Imu(xti::vec3d({NAN, NAN, NAN}), xti::vec3d({NAN, NAN, NAN}))
  {
  }

  Imu(xti::vec3d angular_velocity, xti::vec3d linear_acceleration)
    : m_angular_velocity(angular_velocity)
    , m_linear_acceleration(linear_acceleration)
  {
  }

  xti::vec3d get_angular_velocity() const
  {
    return m_angular_velocity;
  }

  bool has_angular_velocity() const
  {
    return !std::isnan(m_angular_velocity(0));
  }

  xti::vec3d get_linear_acceleration() const
  {
    return m_linear_acceleration;
  }

  bool has_linear_acceleration() const
  {
    return !std::isnan(m_linear_acceleration(0));
  }

private:
  xti::vec3d m_angular_velocity;
  xti::vec3d m_linear_acceleration;
};

template <typename TArchive>
void save(TArchive& archive, const Imu& imu)
{
  archive(imu.get_angular_velocity());
  archive(imu.get_linear_acceleration());
}

template <typename TArchive>
void load(TArchive& archive, Imu& imu)
{
  xti::vec3d angular_velocity;
  archive(angular_velocity);
  xti::vec3d linear_acceleration;
  archive(linear_acceleration);
  imu = Imu(angular_velocity, linear_acceleration);
}

class FrameId
{
public:
  FrameId(std::string dataset_name, std::string location, std::string scene_id, size_t index_in_scene, size_t timestamp, std::shared_ptr<cosy::proj::CRS> crs, cosy::ScaledRigid<double, 3> ego_to_world, cosy::ScaledRigid<double, 2> world_to_crs, std::shared_ptr<cosy::proj::Transformer> epsg4326_to_crs, std::vector<std::shared_ptr<CameraId>> cameras, std::vector<std::shared_ptr<LidarId>> lidars, Imu imu)
    : m_dataset_name(dataset_name)
    , m_location(location)
    , m_scene_id(scene_id)
    , m_index_in_scene(index_in_scene)
    , m_timestamp(timestamp)
    , m_crs(crs)
    , m_ego_to_world(ego_to_world)
    , m_world_to_crs(world_to_crs)
    , m_epsg4326_to_crs(epsg4326_to_crs)
    , m_cameras(cameras)
    , m_lidars(lidars)
    , m_imu(imu)
  {
    if (m_cameras.size() == 0)
    {
      throw std::invalid_argument("Frame must have at least one camera");
    }
  }

  Frame load() const;

  xti::vec2d get_latlon() const
  {
    xti::vec2d latlon = xt::view(m_ego_to_world.get_translation(), xt::range(0, 2));
    latlon = m_world_to_crs.transform(latlon);
    latlon = m_epsg4326_to_crs->transform_inverse(latlon);
    return latlon;
  }

  double get_bearing() const
  {
    xti::vec2d forward_crs = xt::view(xt::linalg::dot(m_ego_to_world.get_rotation(), xti::vec3d({1, 0, 0})), xt::range(0, 2));
    forward_crs = xt::linalg::dot(m_world_to_crs.get_rotation(), forward_crs);
    xti::vec2d north_crs = m_crs->get_axes().get_vector("north");
    return cosy::degrees(cosy::angle(north_crs, forward_crs, true));
  }

  const std::string& get_dataset_name() const
  {
    return m_dataset_name;
  }

  const std::string& get_location() const
  {
    return m_location;
  }

  const std::string& get_scene_id() const
  {
    return m_scene_id;
  }

  size_t get_index_in_scene() const
  {
    return m_index_in_scene;
  }

  size_t get_timestamp() const
  {
    return m_timestamp;
  }

  const std::shared_ptr<cosy::proj::CRS>& get_crs() const
  {
    return m_crs;
  }

  const cosy::ScaledRigid<double, 3>& get_ego_to_world() const
  {
    return m_ego_to_world;
  }

  const cosy::ScaledRigid<double, 2>& get_world_to_crs() const
  {
    return m_world_to_crs;
  }

  const std::shared_ptr<cosy::proj::Transformer>& get_epsg4326_to_crs() const
  {
    return m_epsg4326_to_crs;
  }

  const std::vector<std::shared_ptr<CameraId>>& get_cameras() const
  {
    return m_cameras;
  }

  const std::vector<std::shared_ptr<LidarId>>& get_lidars() const
  {
    return m_lidars;
  }

  const Imu& get_imu() const
  {
    return m_imu;
  }

  std::string get_name() const
  {
    return XTI_TO_STRING(m_dataset_name << "-" << m_scene_id << "-" << std::setw(6) << std::setfill('0') << m_index_in_scene);
  }

  cosy::ScaledRigid<double, 2> get_world_to_pixels_at(xti::vec2d latlon, double bearing, double meters_per_pixel) const
  {
    auto get_old_world_to_pixels_at = [&](xti::vec2d latlon, double bearing){
      cosy::ScaledRigid<double, 2> crsdest_to_crs;
      crsdest_to_crs.get_rotation() = cosy::angle_to_rotation_matrix(this->get_epsg4326_to_crs()->transform_angle(cosy::radians(bearing)));
      crsdest_to_crs.get_translation() = this->get_epsg4326_to_crs()->transform(latlon);

      cosy::ScaledRigid<double, 2> crsdest_to_pixelsdest;
      crsdest_to_pixelsdest.get_scale() = 1.0 / (this->get_world_to_crs().get_scale() * meters_per_pixel);

      cosy::ScaledRigid<double, 2> pixelsdest_to_pixels = cosy::NamedAxesTransformation<double, 2>(this->get_crs()->get_axes(), tiledwebmaps::pixel_axes);

      cosy::ScaledRigid<double, 2> fix_rotation;
      fix_rotation.get_rotation() = cosy::angle_to_rotation_matrix(cosy::radians(-90.0));

      return fix_rotation * pixelsdest_to_pixels * crsdest_to_pixelsdest * crsdest_to_crs.inverse() * this->get_world_to_crs();
    };

    auto own_world_to_pixels = get_old_world_to_pixels_at(this->get_latlon(), this->get_bearing());

    return get_old_world_to_pixels_at(latlon, bearing);
  }

  cosy::ScaledRigid<double, 3> get_ego_to_pixels_at(xti::vec2d latlon, double bearing, double meters_per_pixel) const
  {
    cosy::ScaledRigid<double, 3> ego_to_world_noheight = this->get_ego_to_world();
    ego_to_world_noheight.get_translation()[2] = 0.0;

    cosy::ScaledRigid<double, 2> world_to_pixels = get_world_to_pixels_at(latlon, bearing, meters_per_pixel);

    cosy::ScaledRigid<double, 3> world_to_pixels_3d;
    world_to_pixels_3d.get_scale() = world_to_pixels.get_scale(); // TODO: project 2d to 3d function
    for (int32_t r = 0; r < 2; r++)
    {
      for (int32_t c = 0; c < 2; c++)
      {
        world_to_pixels_3d.get_rotation()(r, c) = world_to_pixels.get_rotation()(r, c);
      }
      world_to_pixels_3d.get_translation()(r) = world_to_pixels.get_translation()(r);
    }

    return world_to_pixels_3d * ego_to_world_noheight;
  }

  template <typename TArchive>
  static void load_and_construct(TArchive& archive, cereal::construct<FrameId>& construct)
  {
    std::string dataset_name;
    archive(dataset_name);
    std::string location;
    archive(location);
    std::string scene_id;
    archive(scene_id);
    size_t index_in_scene;
    archive(index_in_scene);
    size_t timestamp;
    archive(timestamp);
    std::shared_ptr<cosy::proj::CRS> crs;
    archive(crs);
    cosy::ScaledRigid<double, 3> ego_to_world;
    archive(ego_to_world);
    cosy::ScaledRigid<double, 2> world_to_crs;
    archive(world_to_crs);
    std::shared_ptr<cosy::proj::Transformer> epsg4326_to_crs;
    archive(epsg4326_to_crs);

    std::vector<std::shared_ptr<CameraId>> cameras;
    size_t cameras_num;
    archive(cameras_num);
    std::unique_ptr<CameraId> camera_id;
    for (size_t i = 0; i < cameras_num; ++i)
    {
      archive(camera_id);
      cameras.push_back(std::make_shared<CameraId>(*camera_id));
    }

    std::vector<std::shared_ptr<LidarId>> lidars;
    size_t lidars_num;
    archive(lidars_num);
    for (size_t i = 0; i < lidars_num; ++i)
    {
      lidars.push_back(LidarId::load(archive));
    }

    Imu imu;
    archive(imu);

    construct(dataset_name, location, scene_id, index_in_scene, timestamp, crs, ego_to_world, world_to_crs, epsg4326_to_crs, cameras, lidars, imu);
  }

private:
  std::string m_dataset_name;
  std::string m_location;
  std::string m_scene_id;
  size_t m_index_in_scene;
  size_t m_timestamp;
  std::shared_ptr<cosy::proj::CRS> m_crs;
  cosy::ScaledRigid<double, 3> m_ego_to_world;
  cosy::ScaledRigid<double, 2> m_world_to_crs;
  std::shared_ptr<cosy::proj::Transformer> m_epsg4326_to_crs;
  std::vector<std::shared_ptr<CameraId>> m_cameras;
  std::vector<std::shared_ptr<LidarId>> m_lidars;
  Imu m_imu;
};

template <typename TArchive>
void save(TArchive& archive, const FrameId& frame_id)
{
  archive(frame_id.get_dataset_name());
  archive(frame_id.get_location());
  archive(frame_id.get_scene_id());
  archive(frame_id.get_index_in_scene());
  archive(frame_id.get_timestamp());
  archive(frame_id.get_crs());
  archive(frame_id.get_ego_to_world());
  archive(frame_id.get_world_to_crs());
  archive(frame_id.get_epsg4326_to_crs());

  archive(frame_id.get_cameras().size());
  for (auto camera_id : frame_id.get_cameras())
  {
    archive(std::make_unique<CameraId>(*camera_id));
  }
  archive(frame_id.get_lidars().size());
  for (auto lidar_id : frame_id.get_lidars())
  {
    LidarId::save(archive, lidar_id);
  }

  archive(frame_id.get_imu());
}

class AlignedFrameId
{
public:
  AlignedFrameId(FrameId frame_id, xti::vec2d latlon, double bearing, double meters_per_pixel)
    : m_frame_id(frame_id)
    , m_latlon(latlon)
    , m_bearing(bearing)
    , m_meters_per_pixel(meters_per_pixel)
  {
  }

  AlignedFrame load() const;

  const FrameId& get_base_id() const
  {
    return m_frame_id;
  }

  xti::vec2d get_latlon() const
  {
    return m_latlon;
  }

  double get_bearing() const
  {
    return m_bearing;
  }

  double get_meters_per_pixel() const
  {
    return m_meters_per_pixel;
  }

  cosy::ScaledRigid<double, 3> get_ego_to_pixels_at(xti::vec2d latlon, double bearing) const
  {
    return m_frame_id.get_ego_to_pixels_at(latlon, bearing, m_meters_per_pixel);
  }

  cosy::ScaledRigid<double, 3> get_ego_to_pixels() const
  {
    return this->get_ego_to_pixels_at(m_latlon, m_bearing);
  }

private:
  FrameId m_frame_id;
  xti::vec2d m_latlon;
  double m_bearing;
  double m_meters_per_pixel;
  xti::vec2s m_image_shape;
};

class Frame
{
public:
  Frame(FrameId id, std::vector<Camera>&& cameras, xt::xtensor<double, 2>&& points_ego)
    : m_id(id)
    , m_cameras(std::move(cameras))
    , m_points_ego(std::move(points_ego))
  {
  }

  const FrameId& get_id() const
  {
    return m_id;
  }

  const std::vector<Camera>& get_cameras() const
  {
    return m_cameras;
  }

  const xt::xtensor<double, 2>& get_points_ego() const
  {
    return m_points_ego;
  }

private:
  FrameId m_id;
  std::vector<Camera> m_cameras;
  xt::xtensor<double, 2> m_points_ego;
};

Frame FrameId::load() const
{
  std::vector<Lidar> lidars;
  for (const auto& lidar_id : m_lidars)
  {
    lidars.emplace_back(lidar_id->load());
  }
  xt::xtensor<double, 2> points_ego;
  if (lidars.size() > 0)
  {
    points_ego = lidars[0].get_points();
    for (size_t i = 1; i < lidars.size(); i++)
    {
      points_ego = xt::concatenate(xt::xtuple(std::move(points_ego), lidars[i].get_points()), 0);
    }
  }
  else
  {
    points_ego = xt::xtensor<double, 2>({0, 3});
  }

  std::vector<Camera> cameras;
  for (std::shared_ptr<CameraId> camera_id : m_cameras)
  {
    cameras.emplace_back(camera_id->load(points_ego));
  }

  return Frame(*this, std::move(cameras), std::move(points_ego));
}

class AlignedFrame
{
public:
  AlignedFrame(AlignedFrameId id, Frame&& frame, xt::xtensor<double, 2>&& bev_pixels, xti::vec2d vehicle_pixel)
    : m_id(id)
    , m_frame(std::move(frame))
    , m_bev_pixels(std::move(bev_pixels))
    , m_vehicle_pixel(vehicle_pixel)
  {
  }

  const Frame& get_base_frame() const
  {
    return m_frame;
  }

  const AlignedFrameId& get_id() const
  {
    return m_id;
  }

  const xt::xtensor<double, 2>& get_bev_pixels() const
  {
    return m_bev_pixels;
  }

  xti::vec2d get_vehicle_pixel() const
  {
    return m_vehicle_pixel;
  }

private:
  AlignedFrameId m_id;
  Frame m_frame;
  xt::xtensor<double, 2> m_bev_pixels;
  xti::vec2d m_vehicle_pixel;
};

AlignedFrame AlignedFrameId::load() const
{
  Frame frame = m_frame_id.load();

  cosy::ScaledRigid<double, 3> ego_to_pixels = this->get_ego_to_pixels();

  xt::xtensor<double, 2> bev_pixels = xt::xtensor<double, 2>(std::array<size_t, 2>{0, 2});
  if (frame.get_points_ego().shape()[0] > 0)
  {
    bev_pixels = ego_to_pixels.transform_all(frame.get_points_ego());
    bev_pixels = xt::view(std::move(bev_pixels), xt::all(), xt::range(0, 2));
  }

  xti::vec3d vehicle_pixel = ego_to_pixels.transform(xti::vec3d({0, 0, 0}));

  return AlignedFrame(*this, std::move(frame), std::move(bev_pixels), xti::vec2d({vehicle_pixel(0), vehicle_pixel(1)}));
}

} // end of ns georegdata::ground
