#include <pybind11/pybind11.h>
#include <tiledwebmaps/tiledwebmaps.h>
#include <xtensor-python/pytensor.hpp>

namespace py = pybind11;

thread_local std::shared_ptr<cosy::proj::Context> proj_context = std::make_shared<cosy::proj::Context>();

PYBIND11_MODULE(backend, m)
{
  py::class_<tiledwebmaps::Layout, std::shared_ptr<tiledwebmaps::Layout>>(m, "Layout")
    .def(py::init<std::shared_ptr<cosy::proj::CRS>, xti::vec2s, cosy::geo::CompassAxes, bool>(),
      py::arg("crs"),
      py::arg("tile_shape"),
      py::arg("tile_axes"),
      py::arg("use_only_first_bound_axis") = true
    )
    .def("crs_to_tile", &tiledwebmaps::Layout::crs_to_tile,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("tile_to_crs", &tiledwebmaps::Layout::tile_to_crs,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("tile_to_pixel", &tiledwebmaps::Layout::tile_to_pixel,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("pixel_to_tile", &tiledwebmaps::Layout::pixel_to_tile,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("epsg4326_to_tile", &tiledwebmaps::Layout::epsg4326_to_tile,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("tile_to_epsg4326", &tiledwebmaps::Layout::tile_to_epsg4326,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("epsg4326_to_pixel", &tiledwebmaps::Layout::epsg4326_to_pixel,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("pixel_to_epsg4326", &tiledwebmaps::Layout::pixel_to_epsg4326,
      py::arg("coords"),
      py::arg("zoom")
    )
    .def("pixels_per_meter_at_latlon", &tiledwebmaps::Layout::pixels_per_meter_at_latlon,
      py::arg("latlon"),
      py::arg("zoom")
    )
    .def_property_readonly("crs", &tiledwebmaps::Layout::get_crs)
    .def_property_readonly("tile_shape", [](const tiledwebmaps::Layout& layout) -> xti::vec2i {return xt::cast<int>(layout.get_tile_shape());})
    .def_property_readonly("tile_axes", &tiledwebmaps::Layout::get_tile_axes)
    .def_property_readonly("epsg4326_to_crs", &tiledwebmaps::Layout::get_epsg4326_to_crs)
    .def_static("XYZ", [](xti::vec2s tile_shape){
        return tiledwebmaps::Layout::XYZ(proj_context, tile_shape);
      },
      py::arg("tile_shape") = xti::vec2s({256, 256})
    )
    .def_static("TMS", [](xti::vec2s tile_shape){
        return tiledwebmaps::Layout::TMS(proj_context, tile_shape);
      },
      py::arg("tile_shape") = xti::vec2s({256, 256})
    )
  ;

  py::class_<tiledwebmaps::TileLoader, std::shared_ptr<tiledwebmaps::TileLoader>>(m, "TileLoader", py::dynamic_attr())
    .def("load", &tiledwebmaps::TileLoader::load,
      py::arg("tile"),
      py::arg("zoom")
    )
    .def("load", [](tiledwebmaps::TileLoader& tile_loader, xti::vec2s min_tile, xti::vec2s max_tile, size_t zoom){
        return tiledwebmaps::load(tile_loader, min_tile, max_tile, zoom);
      },
      py::arg("min_tile"),
      py::arg("max_tile"),
      py::arg("zoom")
    )
    .def("load", [](tiledwebmaps::TileLoader& tile_loader, xti::vec2d latlon, double bearing, double meters_per_pixel, xti::vec2s shape, size_t zoom){
        return tiledwebmaps::load_metric(tile_loader, latlon, bearing, meters_per_pixel, shape, zoom);
      },
      py::arg("latlon"),
      py::arg("bearing"),
      py::arg("meters_per_pixel"),
      py::arg("shape"),
      py::arg("zoom")
    )
    .def_property_readonly("layout", &tiledwebmaps::TileLoader::get_layout)
  ;

  py::class_<tiledwebmaps::XYZ, std::shared_ptr<tiledwebmaps::XYZ>, tiledwebmaps::TileLoader>(m, "XYZ")
    .def(py::init<std::string, tiledwebmaps::Layout, size_t, float>(),
      py::arg("url"),
      py::arg("layout"),
      py::arg("retries") = 10,
      py::arg("wait_after_error") = 1.5
    )
  ;
  py::class_<tiledwebmaps::Quad, std::shared_ptr<tiledwebmaps::Quad>, tiledwebmaps::TileLoader>(m, "Quad")
    .def(py::init<std::string, tiledwebmaps::Layout, size_t, float>(),
      py::arg("url"),
      py::arg("layout"),
      py::arg("retries") = 10,
      py::arg("wait_after_error") = 1.5
    )
  ;
  py::class_<tiledwebmaps::WMS, std::shared_ptr<tiledwebmaps::WMS>, tiledwebmaps::TileLoader>(m, "WMS")
    .def(py::init<std::string, tiledwebmaps::Layout, size_t, float>(),
      py::arg("url"),
      py::arg("layout"),
      py::arg("retries") = 10,
      py::arg("wait_after_error") = 1.5
    )
  ;

  py::class_<tiledwebmaps::Cache, std::shared_ptr<tiledwebmaps::Cache>, tiledwebmaps::TileLoader>(m, "Cache")
    .def_property_readonly("layout", [](const tiledwebmaps::Cache& cache){return cache.get_layout();})
  ;
  py::class_<tiledwebmaps::CachedTileLoader, std::shared_ptr<tiledwebmaps::CachedTileLoader>, tiledwebmaps::TileLoader>(m, "CachedTileLoader", py::dynamic_attr())
    .def(py::init<std::shared_ptr<tiledwebmaps::TileLoader>, std::shared_ptr<tiledwebmaps::Cache>>())
  ;

  py::class_<tiledwebmaps::Directory, std::shared_ptr<tiledwebmaps::Directory>, tiledwebmaps::Cache>(m, "Directory", py::dynamic_attr())
    .def(py::init<tiledwebmaps::Layout, std::string, std::string, float>(),
      py::arg("layout"),
      py::arg("path"),
      py::arg("format") = "jpg",
      py::arg("wait_after_last_modified") = 1.0
    )
    .def_property_readonly("path", [](const tiledwebmaps::Directory& directory){return directory.get_path().string();})
  ;
  m.def("DirectoryCached", [](std::shared_ptr<tiledwebmaps::TileLoader> loader, std::string path, std::string format, float wait_after_last_modified){
      return std::make_shared<tiledwebmaps::CachedTileLoader>(loader, std::make_shared<tiledwebmaps::Directory>(loader->get_layout(), path, format, wait_after_last_modified));
    },
    py::arg("loader"),
    py::arg("path"),
    py::arg("format") = "jpg",
    py::arg("wait_after_last_modified") = 1.0
  );
  py::class_<tiledwebmaps::LRU, std::shared_ptr<tiledwebmaps::LRU>, tiledwebmaps::Cache>(m, "LRU", py::dynamic_attr())
    .def(py::init<tiledwebmaps::Layout, int>(),
      py::arg("layout"),
      py::arg("size")
    )
  ;
  m.def("LRUCached", [](std::shared_ptr<tiledwebmaps::TileLoader> loader, int size){
      return std::make_shared<tiledwebmaps::CachedTileLoader>(loader, std::make_shared<tiledwebmaps::LRU>(loader->get_layout(), size));
    },
    py::arg("loader"),
    py::arg("size")
  );


  py::register_exception<tiledwebmaps::LoadTileException>(m, "LoadTileException");
  py::register_exception<tiledwebmaps::WriteFileException>(m, "WriteFileException");
  py::register_exception<tiledwebmaps::LoadFileException>(m, "LoadFileException");
  py::register_exception<tiledwebmaps::FileNotFoundException>(m, "FileNotFoundException");
}
