import os, sys, shutil, datetime, yaml, subprocess, time, socket, threading

class LogDir:
    def __init__(self, path, mode=None, environ=["CUDA_VISIBLE_DEVICES"]):
        if not mode in ["d", "delete", "t", "timestamp", "c", "counter"] and not mode is None:
            raise ValueError(f"Invalid mode {mode}")
        use_mode = mode
        while True:
            if use_mode == "d" or use_mode == "delete":
                if os.path.isdir(path):
                    print("Deleting directory " + path)
                    shutil.rmtree(path)
            elif use_mode == "t" or use_mode == "timestamp":
                def get_path_t():
                    return path + "-" + datetime.datetime.now().replace(microsecond=0).isoformat()
                path_t = get_path_t()
                while os.path.isdir(path_t):
                    time.sleep(0.1)
                    path_t = get_path_t()
                path = path_t
            elif use_mode == "c" or use_mode == "counter":
                c = 1
                def path_c():
                    return path + "-" + str(c)
                while os.path.isdir(path_c()):
                    c += 1
                path = path_c()
            elif not use_mode is None:
                print("Invalid option")
                continue

            try:
                os.makedirs(path)
                break
            except FileExistsError as e:
                if mode is None:
                    use_mode = input("Directory " + path + " already exists. Options: Delete (d), append timestamp (t), append counter (c)\n> ")
        assert os.path.isdir(path)
        print(f"Created directory {path}")

        self.path = path

        config = {}

        config["cmd"] = sys.argv
        config["cwd"] = os.getcwd()

        if os.path.isdir(os.path.join(path, ".git")):
            p = subprocess.Popen(["git", "rev-parse", "HEAD"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            out, err = p.communicate()
            if err != 0:
                config["git"] = {}
                config["git"]["commit"] = out.decode().strip()

                patch_file = os.path.join(path, "gitdiff.patch")
                res = os.system("git diff HEAD > " + patch_file)
                if res == 0:
                    config["git"]["diff"] = patch_file

        config["environ"] = {}
        for key in environ:
            if key in os.environ:
                config["environ"][key] = os.environ[key]

        config["server"] = socket.gethostname()

        config["time"] = datetime.datetime.now().replace(microsecond=0).isoformat()

        if "STY" in os.environ:
            config["gnu-screen"] = os.environ["STY"]

        config["pid"] = os.getpid()

        with open(os.path.join(path, "log.yaml"), "w") as f:
            yaml.dump(config, f, default_flow_style=False)

        self.lock = threading.Lock()

    def dir(self, path=None, create=True):
        if path is None:
            return self.path
        if os.path.isabs(path):
            raise ValueError(f"Path must be relative, got {path}")
        path = os.path.join(self.path, path)
        if not os.path.isdir(path):
            with self.lock:
                if os.path.isfile(path):
                    raise FileExistsError(f"Path {path} already exists and is a file")
                if not os.path.isdir(path) and create:
                    os.makedirs(path)
        return path
