from . import stream
import threading, queue, sys, traceback, os

class queued(stream.Stream):
    class WorkerEndTimeout(Exception):
        pass
    class WorkerException(Exception):
        pass

    class StopSignal:
        pass

    def __init__(self, input, workers=1, maxsize=0, end_timeout=30.0, warn_timeout=0, on_worker_error="forward"):
        super().__init__()
        self.input = stream.wrap(input)
        self.maxsize = maxsize
        self.end_timeout = end_timeout
        self.tb = "".join(traceback.format_stack()[:-1])
        self.warn_timeout = warn_timeout
        if not (on_worker_error == "exit" or on_worker_error == "forward"):
            raise ValueError("on_worker_error parameter must be {exit|forward}")
        self.on_worker_error = on_worker_error

        self.q = queue.Queue(maxsize)
        self.force_stop = False

        self.lock = threading.Lock()
        self.exception = None
        self.threads = []
        self.running_workers = workers
        with self.lock:
            for _ in range(workers):
                thread = threading.Thread(target=self.work)
                thread.start()
                self.threads.append(thread)

    def clear_queue(self):
        while not self.q.empty():
            try:
                self.q.get(False)
            except queue.Empty:
                continue
            self.q.task_done()

    def getFill(self):
        return float(min(max(self.q.qsize(), 0), self.maxsize)) / self.maxsize

    def work(self):
        try:
            while not self.force_stop:
                try:
                    item = stream.next(self.input)
                except StopIteration:
                    break
                self.q.put(item)
            if self.force_stop:
                self.clear_queue()
            with self.lock:
                self.running_workers -= 1
                if self.running_workers == 0:
                    self.q.put(queued.StopSignal())
        except:
            if self.on_worker_error == "exit":
                print("Worker got exception:\n" + traceback.format_exc())
                os._exit(-1)
            else:
                self.exception = queued.WorkerException("Worker got exception:\n" + traceback.format_exc())
                while True:
                    try:
                        self.q.put(queued.StopSignal(), block=False)
                        break
                    except queue.Full:
                        self.clear_queue()

    def next(self):
        if not self.exception is None:
            self.stop()
            raise self.exception
        if self.force_stop:
            raise StopIteration
        try:
            item = self.q.get(timeout=self.warn_timeout if self.warn_timeout > 0 else None)
        except queue.Empty:
            print("Waited more than " + str(self.warn_timeout) + " seconds for queued output at:\n" + str(self.tb))
            item = self.q.get()
        if not self.exception is None:
            self.stop()
            raise self.exception
        if isinstance(item, queued.StopSignal):
            self.q.put(queued.StopSignal())
            raise StopIteration
        else:
            return item

    def stop(self):
        self.force_stop = True
        self.clear_queue()
        self.input.stop()
        for thread in self.threads:
            thread.join(self.end_timeout)
            if thread.is_alive():
                raise queued.WorkerEndTimeout("Timeout when waiting for queued worker to finish. Running workers: " + str(self.running_workers))
        while True:
            try:
                self.q.put(queued.StopSignal(), block=False)
                break
            except queue.Full:
                self.clear_queue()
