import math
import numpy as np
from cosy.backend.geo import *

EARTH_RADIUS_METERS = 6.378137e6

def distance(latlon1, latlon2):
    latlon1 = np.radians(np.asarray(latlon1).astype("float64"))
    latlon2 = np.radians(np.asarray(latlon2).astype("float64"))

    a = np.sin((latlon1[..., 0] - latlon2[..., 0]) / 2) ** 2 + np.cos(latlon1[..., 0]) * np.cos(latlon2[..., 0]) * (np.sin((latlon1[..., 1] - latlon2[..., 1]) / 2) ** 2)
    return EARTH_RADIUS_METERS * 2 * np.arctan2(np.sqrt(a), np.sqrt(1 - a))

def bearing(latlon1, latlon2):
    latlon1 = np.radians(np.asarray(latlon1).astype("float64"))
    latlon2 = np.radians(np.asarray(latlon2).astype("float64"))
    dlon = latlon2[..., 1] - latlon1[..., 1]
    x = np.cos(latlon2[..., 0]) * np.sin(dlon)
    y = np.cos(latlon1[..., 0]) * np.sin(latlon2[..., 0]) - np.sin(latlon1[..., 0]) * np.cos(latlon2[..., 0]) * np.cos(dlon)
    return np.degrees(np.arctan2(x, y))

def pixels_per_meter(latlon, zoom, tile_size):
    latlon = np.asarray(latlon).astype("float64")

    center_tile = epsg4326_to_xyz(latlon, zoom).astype("int32") # TODO: convert this to int first? Or -0.5, +0.5 below?
    tile_size_deg = np.absolute(xyz_to_epsg4326(center_tile, zoom) - xyz_to_epsg4326(center_tile + 1, zoom))
    tile_size_meter = tile_size_deg * meters_per_deg(latlon)
    return tile_size / tile_size_meter

# bearing: radians
# distance: meters
def move_from_latlon(latlon, bearing, distance):
    bearing = math.radians(bearing)
    latlon_rad = np.radians(latlon)

    angular_distance = distance / EARTH_RADIUS_METERS

    target_lat = math.asin(
        math.sin(latlon_rad[0]) * math.cos(angular_distance) +
        math.cos(latlon_rad[0]) * math.sin(angular_distance) * math.cos(bearing)
    )
    target_lon = latlon_rad[1] + math.atan2(
        math.sin(bearing) * math.sin(angular_distance) * math.cos(latlon_rad[0]),
        math.cos(angular_distance) - math.sin(latlon_rad[0]) * math.sin(target_lat)
    )
    while target_lon < -math.pi:
        target_lon += 2 * math.pi
    while target_lon > math.pi:
        target_lon -= 2 * math.pi

    return np.array([math.degrees(target_lat), math.degrees(target_lon)])
