import numpy as np
try:
    import g2o
except ImportError:
    pass
import cosy

if "g2o" in globals():
    class G2O(g2o.SparseOptimizer):
        def __init__(self):
            super().__init__()
            solver = g2o.BlockSolverSE2(g2o.LinearSolverCholmodSE2())
            solver = g2o.OptimizationAlgorithmLevenberg(solver)
            super().set_algorithm(solver)

            self.next_vertex_id = 0
            self.frame_ids = []
            self.frame_to_world = []

        def optimize(self, max_iterations=100, verbose=False):
            super().initialize_optimization()
            if verbose:
                super().set_verbose(True)
            its = super().optimize(max_iterations)
            return its

        def add_frame(self, frame_to_world, gps_to_world_translation=None, gps_confidence=1.0):
            def add_vertex(pose, fixed):
                v_se2 = g2o.VertexSE2()
                v_se2.set_id(self.next_vertex_id)
                v_se2.set_estimate(g2o.SE2(g2o.Isometry2d(pose.to_matrix())))
                v_se2.set_fixed(fixed)
                self.add_vertex(v_se2)
                self.next_vertex_id += 1
                return self.next_vertex_id - 1

            def add_edge(v1, v2, transform, information=np.identity(3), robust_kernel=g2o.RobustKernelHuber(np.sqrt(5.992))): # g2o.RobustKernelDCS()):
                edge = g2o.EdgeSE2()
                edge.set_vertex(0, self.vertex(v1))
                edge.set_vertex(1, self.vertex(v2))

                edge.set_measurement(g2o.SE2(g2o.Isometry2d(transform.to_matrix())))
                edge.set_information(information)
                if robust_kernel is not None:
                    edge.set_robust_kernel(robust_kernel)
                self.add_edge(edge)



            # if gps_to_world_translation is None:
            #     frame_id = add_vertex(frame_to_world, fixed=False)
            #     # add_edge(frame_id, add_vertex(frame_to_world, fixed=True), cosy.np.Rigid(dtype="float64", rank=2), information=np.diag([1.0, 1e-5, 1e-5]))
            # else:
            #     frame_id = add_vertex(cosy.np.Rigid(
            #         translation=gps_to_world_translation,
            #         rotation=frame_to_world.rotation,
            #     ), fixed=True)

            frame_id = add_vertex(frame_to_world, fixed=False)

            for offset in [1]: # , 5, 10, 15, 20, 30, 50
                if len(self.frame_to_world) >= offset:
                    add_edge(self.frame_ids[-offset], frame_id, self.frame_to_world[-offset].inverse() * frame_to_world, information=np.diag([10.0, 10.0, 10.0]))

            if not gps_to_world_translation is None:
                gps_id = add_vertex(cosy.np.Rigid(
                    translation=gps_to_world_translation,
                    rotation=frame_to_world.rotation,
                ), fixed=True)
                add_edge(frame_id, gps_id, cosy.np.Rigid(dtype="float64", rank=2), information=np.diag([gps_confidence, gps_confidence, 1.0]))

            self.frame_ids.append(frame_id)
            self.frame_to_world.append(frame_to_world)

        def get_frame_to_world(self):
            return [cosy.np.Rigid.from_matrix(self.vertex(id).estimate().to_isometry().matrix()) for id in self.frame_ids]
