#pragma once

#include <cosy/affine.h>
#include <cosy/proj.h>
#include <xtensor/xtensor.hpp>
#include <xtensor-blas/xlinalg.hpp>
#include <tiledwebmaps/tiledwebmaps.h>

namespace georegdata::aerial {

class Frame;

class FrameId
{
public:
  FrameId(std::shared_ptr<tiledwebmaps::TileLoader> tile_loader, std::string tile_loader_name, size_t zoom, xti::vec2d latlon, double bearing, double meters_per_pixel, xti::vec2s image_shape)
    : m_tile_loader(tile_loader)
    , m_tile_loader_name(tile_loader_name)
    , m_zoom(zoom)
    , m_latlon(latlon)
    , m_bearing(bearing)
    , m_meters_per_pixel(meters_per_pixel)
    , m_image_shape(image_shape)
  {
  }

  Frame load() const;

  const std::shared_ptr<tiledwebmaps::TileLoader>& get_tile_loader() const
  {
    return m_tile_loader;
  }

  const size_t& get_zoom() const
  {
    return m_zoom;
  }

  const xti::vec2d& get_latlon() const
  {
    return m_latlon;
  }

  const double& get_bearing() const
  {
    return m_bearing;
  }

  const double& get_meters_per_pixel() const
  {
    return m_meters_per_pixel;
  }

  const xti::vec2s& get_image_shape() const
  {
    return m_image_shape;
  }

  std::string get_name() const
  {
    return XTI_TO_STRING(m_tile_loader_name << "-zoom" << m_zoom);
  }

private:
  std::shared_ptr<tiledwebmaps::TileLoader> m_tile_loader;
  std::string m_tile_loader_name;
  size_t m_zoom;
  xti::vec2d m_latlon;
  double m_bearing;
  double m_meters_per_pixel;
  xti::vec2s m_image_shape;
};

class Frame
{
public:
  Frame(FrameId id, xt::xtensor<double, 3>&& image)
    : m_id(id)
    , m_image(std::move(image))
  {
  }

  const FrameId& get_id() const
  {
    return m_id;
  }

  const xt::xtensor<double, 3>& get_image() const
  {
    return m_image;
  }

private:
  FrameId m_id;
  xt::xtensor<double, 3> m_image;
};

Frame FrameId::load() const
{
  // Image is loaded with bearing pointing upwards in image, but we want it to point in positive first axis direction
  xt::xtensor<double, 3> image = tiledwebmaps::load_metric(*m_tile_loader, m_latlon, m_bearing + 180.0, m_meters_per_pixel, m_image_shape, m_zoom);

  return Frame(*this, std::move(image));
}

} // end of ns georegdata::aerial
