#pragma once

#include "aerial/frame.h"
#include "ground/frame.h"

namespace georegdata {

class Frame;

class FrameId
{
public:
  FrameId(ground::AlignedFrameId ground_frame_id, aerial::FrameId aerial_frame_id)
    : m_ground_frame_id(ground_frame_id)
    , m_aerial_frame_id(aerial_frame_id)
  {
  }

  Frame load() const;

  const ground::AlignedFrameId& get_ground_frame_id() const
  {
    return m_ground_frame_id;
  }

  const aerial::FrameId& get_aerial_frame_id() const
  {
    return m_aerial_frame_id;
  }

  std::string get_name() const
  {
    return XTI_TO_STRING(m_ground_frame_id.get_base_id().get_name() << "-" << m_aerial_frame_id.get_name());
  }

  cosy::ScaledRigid<double, 2> get_bevmeters_to_aerialmeters() const
  {
    const auto& epsg4326_to_crs = *this->get_ground_frame_id().get_base_id().get_epsg4326_to_crs();
    const auto& crs = *this->get_ground_frame_id().get_base_id().get_crs();
    double crs_scale = this->get_ground_frame_id().get_base_id().get_world_to_crs().get_scale();

    cosy::ScaledRigid<double, 2> crsaxes_to_pixelaxes = cosy::NamedAxesTransformation<double, 2>(crs.get_axes(), tiledwebmaps::pixel_axes);

    cosy::ScaledRigid<double, 2> unscaledbev_to_crs;
    unscaledbev_to_crs.get_rotation() = cosy::angle_to_rotation_matrix(epsg4326_to_crs.transform_angle(cosy::radians(m_ground_frame_id.get_bearing())));
    unscaledbev_to_crs.get_translation() = epsg4326_to_crs.transform(m_ground_frame_id.get_latlon());

    cosy::ScaledRigid<double, 2> unscaledbev_to_bev;
    unscaledbev_to_bev.get_scale() = 1.0 / crs_scale;

    cosy::ScaledRigid<double, 2> unscaledaerial_to_crs;
    unscaledaerial_to_crs.get_rotation() = cosy::angle_to_rotation_matrix(epsg4326_to_crs.transform_angle(cosy::radians(m_aerial_frame_id.get_bearing())));
    unscaledaerial_to_crs.get_translation() = epsg4326_to_crs.transform(m_aerial_frame_id.get_latlon());

    cosy::ScaledRigid<double, 2> unscaledaerial_to_aerial;
    unscaledaerial_to_aerial.get_scale() = 1.0 / crs_scale;

    cosy::ScaledRigid<double, 2> fix_rotation;
    fix_rotation.get_rotation() = cosy::angle_to_rotation_matrix(cosy::radians(-90.0));

    return fix_rotation * crsaxes_to_pixelaxes * unscaledaerial_to_aerial * unscaledaerial_to_crs.inverse() * unscaledbev_to_crs * unscaledbev_to_bev.inverse() * crsaxes_to_pixelaxes.inverse() * fix_rotation.inverse();
  }

  cosy::ScaledRigid<double, 2> get_bevpixels_to_aerialpixels(double bev_meters_per_pixel, double aerial_meters_per_pixel) const
  {
    cosy::ScaledRigid<double, 2> bevpixels_to_bevmeters;
    bevpixels_to_bevmeters.get_scale() = bev_meters_per_pixel;
    cosy::ScaledRigid<double, 2> aerialpixels_to_aerialmeters;
    aerialpixels_to_aerialmeters.get_scale() = aerial_meters_per_pixel;

    return aerialpixels_to_aerialmeters.inverse() * get_bevmeters_to_aerialmeters() * bevpixels_to_bevmeters;
  }

  cosy::ScaledRigid<double, 2> get_bevpixels_to_aerialpixels() const
  {
    return get_bevpixels_to_aerialpixels(m_ground_frame_id.get_meters_per_pixel(), m_aerial_frame_id.get_meters_per_pixel());
  }

private:
  ground::AlignedFrameId m_ground_frame_id;
  aerial::FrameId m_aerial_frame_id;
};

class Frame
{
public:
  Frame(FrameId id, ground::AlignedFrame&& ground_frame, aerial::Frame&& aerial_frame)
    : m_id(id)
    , m_ground_frame(std::move(ground_frame))
    , m_aerial_frame(std::move(aerial_frame))
  {
  }

  const FrameId& get_id() const
  {
    return m_id;
  }

  const ground::AlignedFrame& get_ground_frame() const
  {
    return m_ground_frame;
  }

  const aerial::Frame& get_aerial_frame() const
  {
    return m_aerial_frame;
  }

private:
  FrameId m_id;
  ground::AlignedFrame m_ground_frame;
  aerial::Frame m_aerial_frame;
};

Frame FrameId::load() const
{
  return Frame(*this, ground::AlignedFrame(m_ground_frame_id.load()), aerial::Frame(m_aerial_frame_id.load()));
}

} // end of ns georegdata
