#include <cereal/archives/binary.hpp>
#include <xtensor-python/pytensor.hpp>
#include <pybind11/pybind11.h>
#include <cosy/python.h>
#include <pybind11/stl.h>
#include <xti/cereal.h>
#include <cereal/types/memory.hpp>
#include <sstream>
#include <pybind11/stl_bind.h>

#include "ground/frame.h"
#include "aerial/frame.h"
#include "frame.h"

namespace py = pybind11;

PYBIND11_MODULE(backend, m)
{
  auto aerial = m.def_submodule("aerial");
  auto ground = m.def_submodule("ground");

  py::class_<georegdata::ground::Imu>(ground, "Imu")
    .def(py::init([](xti::vec3d angular_velocity, xti::vec3d linear_acceleration){
        return georegdata::ground::Imu(angular_velocity, linear_acceleration);
      }),
      py::arg("angular_velocity") = xti::vec3d({NAN, NAN, NAN}),
      py::arg("linear_acceleration") = xti::vec3d({NAN, NAN, NAN})
    )
    .def("has_angular_velocity", &georegdata::ground::Imu::has_angular_velocity)
    .def_property_readonly("angular_velocity", &georegdata::ground::Imu::get_angular_velocity)
    .def("has_linear_acceleration", &georegdata::ground::Imu::has_linear_acceleration)
    .def_property_readonly("linear_acceleration", &georegdata::ground::Imu::get_linear_acceleration)
  ;

  py::class_<georegdata::ground::Camera, std::shared_ptr<georegdata::ground::Camera>>(ground, "Camera")
    .def_property_readonly("image", &georegdata::ground::Camera::get_image)
    .def_property_readonly("image_mask", &georegdata::ground::Camera::get_image_mask)
    .def_property_readonly("pixels", &georegdata::ground::Camera::get_pixels)
    .def_property_readonly("points_depth", &georegdata::ground::Camera::get_points_depth)
    .def_property_readonly("points_mask", &georegdata::ground::Camera::get_points_mask)
    .def_property_readonly("name", [](const georegdata::ground::Camera& camera){return camera.get_id().get_name();})
    .def_property_readonly("intr", [](const georegdata::ground::Camera& camera){return camera.get_id().get_projection();})
    .def_property_readonly("resolution", [](const georegdata::ground::Camera& camera){return camera.get_id().get_resolution();})
    .def_property_readonly("ego_to_camera", [](const georegdata::ground::Camera& camera){return camera.get_id().get_ego_to_camera();})
  ;
  py::class_<georegdata::ground::CameraId, std::shared_ptr<georegdata::ground::CameraId>>(ground, "CameraId")
    .def(py::init([](std::string name, cosy::PinholeK<double, 3> projection, xti::vec2s resolution, cosy::Rigid<double, 3> ego_to_camera, std::string image_file){
        return georegdata::ground::CameraId(name, projection, resolution, ego_to_camera, image_file);
      }),
      py::arg("name"),
      py::arg("intr"),
      py::arg("resolution"),
      py::arg("ego_to_camera"),
      py::arg("image_file")
    )
    .def("load_resolution", &georegdata::ground::CameraId::load_resolution)
    .def_property_readonly("name", &georegdata::ground::CameraId::get_name)
    .def_property_readonly("intr", &georegdata::ground::CameraId::get_projection)
    .def_property_readonly("resolution", &georegdata::ground::CameraId::get_resolution)
    .def_property_readonly("ego_to_camera", &georegdata::ground::CameraId::get_ego_to_camera)
    .def_property_readonly("image_file", &georegdata::ground::CameraId::get_image_file)
  ;
  py::class_<georegdata::ground::AugmentedCameraId, std::shared_ptr<georegdata::ground::AugmentedCameraId>, georegdata::ground::CameraId>(ground, "AugmentedCameraId")
    .def(py::init([](georegdata::ground::CameraId camera_id, double angle, double scale, xti::vec2s max_shape){
        return georegdata::ground::AugmentedCameraId(camera_id, angle, scale, max_shape);
      }),
      py::arg("camera_id"),
      py::arg("angle"),
      py::arg("scale"),
      py::arg("max_shape")
    )
    .def(py::init([](georegdata::ground::CameraId camera_id, double angle, double scale){
        return georegdata::ground::AugmentedCameraId(camera_id, angle, scale);
      }),
      py::arg("camera_id"),
      py::arg("angle"),
      py::arg("scale")
    )
  ;
  py::class_<georegdata::ground::FixedIntrCameraId, std::shared_ptr<georegdata::ground::FixedIntrCameraId>, georegdata::ground::CameraId>(ground, "FixedIntrCameraId")
    .def(py::init([](georegdata::ground::CameraId camera_id, double focal_length, xti::vec2s max_shape){
        return georegdata::ground::FixedIntrCameraId(camera_id, focal_length, max_shape);
      }),
      py::arg("camera_id"),
      py::arg("focal_length"),
      py::arg("max_shape")
    )
    .def(py::init([](georegdata::ground::CameraId camera_id, double focal_length){
        return georegdata::ground::FixedIntrCameraId(camera_id, focal_length);
      }),
      py::arg("camera_id"),
      py::arg("focal_length")
    )
  ;

  py::class_<georegdata::ground::Lidar, std::shared_ptr<georegdata::ground::Lidar>>(ground, "Lidar")
  ;
  py::class_<georegdata::ground::LidarId, std::shared_ptr<georegdata::ground::LidarId>>(ground, "LidarId")
    .def_property_readonly("name", &georegdata::ground::LidarId::get_name)
  ;
  py::class_<georegdata::ground::NpzLidarId, std::shared_ptr<georegdata::ground::NpzLidarId>, georegdata::ground::LidarId>(ground, "NpzLidarId")
    .def(py::init([](std::string name, std::string file, xt::xtensor<double, 2> map, cosy::Rigid<double, 3> loaded_to_ego){
        return georegdata::ground::NpzLidarId(name, file, map, loaded_to_ego);
      }),
      py::arg("name"),
      py::arg("file"),
      py::arg("map") = (xt::xtensor<double, 2>() = xt::eye<double>(3)),
      py::arg("loaded_to_ego") = cosy::Rigid<double, 3>()
    )
    .def_property_readonly("loaded_to_ego", &georegdata::ground::NpzLidarId::get_loaded_to_ego)
    .def_property_readonly("file", &georegdata::ground::NpzLidarId::get_file)
  ;
  py::class_<georegdata::ground::BinLidarId<float>, std::shared_ptr<georegdata::ground::BinLidarId<float>>, georegdata::ground::LidarId>(ground, "Bin32LidarId")
    .def(py::init([](std::string name, std::string file, xt::xtensor<double, 2> map, cosy::Rigid<double, 3> loaded_to_ego, bool crop_to_size){
        return georegdata::ground::BinLidarId<float>(name, file, map, loaded_to_ego, crop_to_size);
      }),
      py::arg("name"),
      py::arg("file"),
      py::arg("map") = (xt::xtensor<double, 2>() = xt::eye<double>(3)),
      py::arg("loaded_to_ego") = cosy::Rigid<double, 3>(),
      py::arg("crop_to_size") = false
    )
    .def_property_readonly("loaded_to_ego", &georegdata::ground::BinLidarId<float>::get_loaded_to_ego)
    .def_property_readonly("file", &georegdata::ground::BinLidarId<float>::get_file)
  ;
  py::class_<georegdata::ground::DummyLidarId, std::shared_ptr<georegdata::ground::DummyLidarId>, georegdata::ground::LidarId>(ground, "DummyLidarId")
    .def(py::init([](std::string name){
        return georegdata::ground::DummyLidarId(name);
      }),
      py::arg("name")
    )
  ;

  py::class_<georegdata::ground::Frame, std::shared_ptr<georegdata::ground::Frame>>(ground, "Frame")
    .def_property_readonly("frame_id", [](const georegdata::ground::Frame& frame){return frame.get_id();})
    .def_property_readonly("cameras", &georegdata::ground::Frame::get_cameras)
    .def_property_readonly("points_ego", [](const georegdata::ground::Frame& frame){return frame.get_points_ego();})
    .def_property_readonly("name", [](const georegdata::ground::Frame& frame){return frame.get_id().get_name();})
    .def_property_readonly("crs", [](const georegdata::ground::Frame& frame){return frame.get_id().get_crs();})
    .def_property_readonly("ego_to_world", [](const georegdata::ground::Frame& frame){return frame.get_id().get_ego_to_world();})
    .def_property_readonly("world_to_crs", [](const georegdata::ground::Frame& frame){return frame.get_id().get_world_to_crs();})
    .def_property_readonly("epsg4326_to_crs", [](const georegdata::ground::Frame& frame){return frame.get_id().get_epsg4326_to_crs();})
    .def_property_readonly("dataset_name", [](const georegdata::ground::Frame& frame){return frame.get_id().get_dataset_name();})
  ;
  py::class_<georegdata::ground::FrameId, std::shared_ptr<georegdata::ground::FrameId>>(ground, "FrameId")
    .def(py::init([](std::string dataset_name, std::string location, std::string scene_id, size_t index_in_scene, size_t timestamp, std::shared_ptr<cosy::proj::CRS> crs, cosy::ScaledRigid<double, 3> ego_to_world, cosy::ScaledRigid<double, 2> world_to_crs, std::shared_ptr<cosy::proj::Transformer> epsg4326_to_crs, std::vector<std::shared_ptr<georegdata::ground::CameraId>> cameras, std::vector<std::shared_ptr<georegdata::ground::LidarId>> lidars, georegdata::ground::Imu imu){
        return georegdata::ground::FrameId(dataset_name, location, scene_id, index_in_scene, timestamp, crs, ego_to_world, world_to_crs, epsg4326_to_crs, cameras, lidars, imu);
      }),
      py::arg("dataset_name"),
      py::arg("location"),
      py::arg("scene_id"),
      py::arg("index_in_scene"),
      py::arg("timestamp"),
      py::arg("crs"),
      py::arg("ego_to_world"),
      py::arg("world_to_crs"),
      py::arg("epsg4326_to_crs"),
      py::arg("cameras"),
      py::arg("lidars"),
      py::arg("imu") = georegdata::ground::Imu()
    )
    .def("load", [](const georegdata::ground::FrameId& frame_id){
        py::gil_scoped_release gil;
        return frame_id.load();
      }
    )
    .def_property_readonly("latlon", &georegdata::ground::FrameId::get_latlon)
    .def_property_readonly("bearing", &georegdata::ground::FrameId::get_bearing)
    .def_property_readonly("dataset_name", &georegdata::ground::FrameId::get_dataset_name)
    .def_property_readonly("location", &georegdata::ground::FrameId::get_location)
    .def_property_readonly("scene_id", &georegdata::ground::FrameId::get_scene_id)
    .def_property_readonly("index_in_scene", &georegdata::ground::FrameId::get_index_in_scene)
    .def_property_readonly("timestamp", &georegdata::ground::FrameId::get_timestamp)
    .def_property_readonly("crs", &georegdata::ground::FrameId::get_crs)
    .def_property_readonly("ego_to_world", &georegdata::ground::FrameId::get_ego_to_world)
    .def_property_readonly("world_to_crs", &georegdata::ground::FrameId::get_world_to_crs)
    .def_property_readonly("epsg4326_to_crs", &georegdata::ground::FrameId::get_epsg4326_to_crs)
    .def_property_readonly("cameras", &georegdata::ground::FrameId::get_cameras)
    .def_property_readonly("lidars", &georegdata::ground::FrameId::get_lidars)
    .def_property_readonly("imu", &georegdata::ground::FrameId::get_imu)
    .def_property_readonly("name", &georegdata::ground::FrameId::get_name)
    .def(py::pickle(
      [](std::shared_ptr<georegdata::ground::FrameId> frame_id) { // __getstate__
        std::ostringstream stream;
        {
          py::gil_scoped_release gil;
          cereal::BinaryOutputArchive archive(stream);
          archive(frame_id);
        }
        return py::bytes(stream.str());
      },
      [](py::bytes data) { // __setstate__
        std::istringstream stream;
        stream.str(data.cast<std::string>());

        py::gil_scoped_release gil;
        cereal::BinaryInputArchive archive(stream);
        std::shared_ptr<georegdata::ground::FrameId> frame_id;
        archive(frame_id);
        return frame_id;
      }
    ))
    .def("unpickle_scenes", [](py::bytes data){
        std::istringstream stream;
        stream.str(data.cast<std::string>());

        py::gil_scoped_release gil;
        cereal::BinaryInputArchive archive(stream);

        std::vector<std::vector<std::shared_ptr<georegdata::ground::FrameId>>> scenes;
        size_t scenes_num;
        archive(scenes_num);

        for (size_t j = 0; j < scenes_num; ++j)
        {
          std::vector<std::shared_ptr<georegdata::ground::FrameId>> scene;
          size_t frames_num;
          archive(frames_num);
          std::unique_ptr<georegdata::ground::FrameId> frame_id;
          for (size_t i = 0; i < frames_num; ++i)
          {
            archive(frame_id);
            scene.push_back(std::make_shared<georegdata::ground::FrameId>(*frame_id));
          }
          scenes.push_back(scene);
        }

        return scenes;
      }
    )
    .def("pickle_scenes", [](std::vector<std::vector<std::shared_ptr<georegdata::ground::FrameId>>> scenes){
        std::ostringstream stream;
        {
          py::gil_scoped_release gil;
          cereal::BinaryOutputArchive archive(stream);
          archive(scenes.size());
          for (auto& scene : scenes)
          {
            archive(scene.size());
            for (auto frame_id : scene)
            {
              archive(std::make_unique<georegdata::ground::FrameId>(*frame_id));
            }
          }
        }
        return py::bytes(stream.str());
      }
    )
  ;
  py::bind_vector<std::vector<std::shared_ptr<georegdata::ground::FrameId>>>(m, "GroundFrameIdVector");

  py::class_<georegdata::ground::AlignedFrame, std::shared_ptr<georegdata::ground::AlignedFrame>>(ground, "AlignedFrame")
    .def_property_readonly("frame_id", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id();})
    .def_property_readonly("cameras", [](const georegdata::ground::AlignedFrame& frame){return frame.get_base_frame().get_cameras();})
    .def_property_readonly("bev_pixels", [](const georegdata::ground::AlignedFrame& frame){return frame.get_bev_pixels();})
    .def_property_readonly("points_ego", [](const georegdata::ground::AlignedFrame& frame){return frame.get_base_frame().get_points_ego();})
    .def_property_readonly("points_num", [](const georegdata::ground::AlignedFrame& frame){return frame.get_base_frame().get_points_ego().shape()[0];})
    .def_property_readonly("name", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_name();})
    .def_property_readonly("vehicle_pixel", [](const georegdata::ground::AlignedFrame& frame){return frame.get_vehicle_pixel();})
    .def_property_readonly("ego_to_pixels", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_ego_to_pixels();})
    .def_property_readonly("epsg4326_to_crs", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_epsg4326_to_crs();})
    .def_property_readonly("ego_to_world", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_ego_to_world();})
    .def_property_readonly("world_to_crs", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_world_to_crs();})
    .def_property_readonly("latlon", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_latlon();})
    .def_property_readonly("bearing", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_bearing();})
    .def_property_readonly("dataset_name", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_dataset_name();})
    .def_property_readonly("location", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_location();})
    .def_property_readonly("scene_id", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_scene_id();})
    .def_property_readonly("index_in_scene", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_index_in_scene();})
    .def_property_readonly("imu", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_imu();})
    .def_property_readonly("timestamp", [](const georegdata::ground::AlignedFrame& frame){return frame.get_id().get_base_id().get_timestamp();})
  ;
  py::class_<georegdata::ground::AlignedFrameId, std::shared_ptr<georegdata::ground::AlignedFrameId>>(ground, "AlignedFrameId")
    .def(py::init([](georegdata::ground::FrameId frame_id, xti::vec2d latlon, double bearing, double meters_per_pixel){
        return georegdata::ground::AlignedFrameId(std::move(frame_id), latlon, bearing, meters_per_pixel);
      }),
      py::arg("frame_id"),
      py::arg("latlon"),
      py::arg("bearing"),
      py::arg("meters_per_pixel")
    )
    .def("load", [](const georegdata::ground::AlignedFrameId& frame_id){
      py::gil_scoped_release gil;
      return frame_id.load();
    })
    .def_property_readonly("base_frame_id", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id();})
    .def_property_readonly("name", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_name();})
    .def_property_readonly("ego_to_pixels", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_ego_to_pixels();})
    .def_property_readonly("epsg4326_to_crs", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_epsg4326_to_crs();})
    .def_property_readonly("ego_to_world", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_ego_to_world();})
    .def_property_readonly("world_to_crs", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_world_to_crs();})
    .def_property_readonly("latlon", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_latlon();})
    .def_property_readonly("bearing", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_bearing();})
    .def_property_readonly("meters_per_pixel", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_meters_per_pixel();})
    .def_property_readonly("dataset_name", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_dataset_name();})
    .def_property_readonly("location", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_location();})
    .def_property_readonly("scene_id", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_scene_id();})
    .def_property_readonly("index_in_scene", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_index_in_scene();})
    .def_property_readonly("imu", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_imu();})
    .def_property_readonly("timestamp", [](const georegdata::ground::AlignedFrameId& frame){return frame.get_base_id().get_timestamp();})
  ;

  py::class_<georegdata::aerial::Frame, std::shared_ptr<georegdata::aerial::Frame>>(aerial, "Frame")
    .def_property_readonly("frame_id", [](const georegdata::aerial::Frame& frame){return frame.get_id();})
    .def_property_readonly("image", &georegdata::aerial::Frame::get_image)
    .def_property_readonly("name", [](const georegdata::aerial::Frame& frame){return frame.get_id().get_name();})
    .def_property_readonly("layout", [](const georegdata::aerial::Frame& frame){return frame.get_id().get_tile_loader()->get_layout();})
    .def_property_readonly("latlon", [](const georegdata::aerial::Frame& frame){return frame.get_id().get_latlon();})
    .def_property_readonly("bearing", [](const georegdata::aerial::Frame& frame){return frame.get_id().get_bearing();})
  ;
  py::class_<georegdata::aerial::FrameId, std::shared_ptr<georegdata::aerial::FrameId>>(aerial, "FrameId")
    .def(py::init([](std::shared_ptr<tiledwebmaps::TileLoader> tile_loader, std::string tile_loader_name, size_t zoom, xti::vec2d latlon, double bearing, double meters_per_pixel, xti::vec2s shape){
        return georegdata::aerial::FrameId(tile_loader, tile_loader_name, zoom, latlon, bearing, meters_per_pixel, shape);
      }),
      py::arg("tile_loader"),
      py::arg("tile_loader_name"),
      py::arg("zoom"),
      py::arg("latlon"),
      py::arg("bearing"),
      py::arg("meters_per_pixel"),
      py::arg("shape")
    )
    .def("load", [](const georegdata::aerial::FrameId& frame_id){
      py::gil_scoped_release gil;
      return frame_id.load();
    })
    .def_property_readonly("name", [](const georegdata::aerial::FrameId& frame){return frame.get_name();})
    .def_property_readonly("layout", [](const georegdata::aerial::FrameId& frame){return frame.get_tile_loader()->get_layout();})
    .def_property_readonly("latlon", [](const georegdata::aerial::FrameId& frame){return frame.get_latlon();})
    .def_property_readonly("bearing", [](const georegdata::aerial::FrameId& frame){return frame.get_bearing();})
  ;

  py::class_<georegdata::Frame, std::shared_ptr<georegdata::Frame>>(m, "Frame")
    .def(py::init([](const georegdata::ground::AlignedFrame& ground_frame, const georegdata::aerial::Frame& aerial_frame){
        return georegdata::Frame(
          georegdata::FrameId(ground_frame.get_id(), aerial_frame.get_id()),
          georegdata::ground::AlignedFrame(ground_frame),
          georegdata::aerial::Frame(aerial_frame)
        );
      }),
      py::arg("ground_frame"),
      py::arg("aerial_frame")
    )
    .def_property_readonly("ground_frame", &georegdata::Frame::get_ground_frame)
    .def_property_readonly("aerial_frame", &georegdata::Frame::get_aerial_frame)
    .def_property_readonly("name", [](const georegdata::Frame& frame){return frame.get_id().get_name();})
    .def_property_readonly("bevmeters_to_aerialmeters", [](const georegdata::Frame& frame){return frame.get_id().get_bevmeters_to_aerialmeters();})
    .def_property_readonly("bevpixels_to_aerialpixels", [](const georegdata::Frame& frame){return frame.get_id().get_bevpixels_to_aerialpixels();})
    .def_property_readonly("image_shape", [](const georegdata::Frame& frame){return frame.get_id().get_aerial_frame_id().get_image_shape();})
  ;
  py::class_<georegdata::FrameId, std::shared_ptr<georegdata::FrameId>>(m, "FrameId")
    .def(py::init([](const georegdata::ground::AlignedFrameId& ground_frame_id, const georegdata::aerial::FrameId& aerial_frame_id){
        return georegdata::FrameId(ground_frame_id, aerial_frame_id);
      }),
      py::arg("ground_frame_id"),
      py::arg("aerial_frame_id")
    )
    .def("load", [](const georegdata::FrameId& frame_id){
      py::gil_scoped_release gil;
      return frame_id.load();
    })
    .def_property_readonly("ground_frame_id", [](const georegdata::FrameId& frame){return frame.get_ground_frame_id();})
    .def_property_readonly("aerial_frame_id", [](const georegdata::FrameId& frame){return frame.get_aerial_frame_id();})
    .def_property_readonly("name", [](const georegdata::FrameId& frame){return frame.get_name();})
    .def_property_readonly("bevmeters_to_aerialmeters", [](const georegdata::FrameId& frame){return frame.get_bevmeters_to_aerialmeters();})
    .def_property_readonly("bevpixels_to_aerialpixels", [](const georegdata::FrameId& frame){return frame.get_bevpixels_to_aerialpixels();})
    .def_property_readonly("image_shape", [](const georegdata::FrameId& frame){return frame.get_aerial_frame_id().get_image_shape();})
  ;
}
