import os, pyunpack, tfcv, tqdm, imageio, requests, tarfile, cosy, cv2
import numpy as np
import tinypl as pl

def compute_bearings_for_latlons(latlons, min_distance_for_bearing=2.0): # meters
    bearings = []
    for i in range(len(latlons)):
        for d in range(len(latlons)):
            i0 = max(i - d, 0)
            i1 = min(i + d, len(latlons) - 1)
            if cosy.np.geo.distance(latlons[i0], latlons[i1]) > min_distance_for_bearing:
                bearings.append(cosy.np.geo.bearing(latlons[i0], latlons[i1]))
                break
        else:
            raise ValueError(f"Failed to compute bearing in trajectory from {latlons[0]} to {latlons[1]}, path length is below minimum distance")
    assert len(bearings) == len(latlons)
    return np.asarray(bearings)

def download(url, file, retries=100, timeout=10.0):
    dir = os.path.dirname(os.path.abspath(file))
    if not os.path.isdir(dir):
        os.makedirs(dir)

    for _ in range(retries):
        if os.path.isfile(file):
            os.remove(file)
        elif os.path.isdir(file):
            raise ValueError("Target path is a directory")
        try:
            resp = requests.get(url, stream=True, timeout=timeout)
            total = int(resp.headers.get("content-length", 0))
            received = 0
            with open(file, "wb") as f, tqdm.tqdm(desc="Download " + url.split("/")[-1], total=total, unit="iB", unit_scale=True, unit_divisor=1024) as bar:
                for data in resp.iter_content(chunk_size=1024):
                    size = f.write(data)
                    bar.update(size)
                    received += size
            if received < total:
                error = requests.exceptions.RequestException("Content too short", response=resp)
                continue
        except requests.exceptions.RequestException as e:
            error = e
            continue
        break
    else:
        if os.path.isfile(file):
            os.remove(file)
        raise error

def extract(src, dest=None):
    if dest is None:
        dest = os.path.dirname(src)
    print(f"Extracting {src}")
    if not os.path.isdir(dest):
        os.makedirs(dest)
    if src.endswith(".tar"):
        tar = tarfile.open(src)
        tar.extractall(path=dest)
        tar.close()
    else:
        pyunpack.Archive(src).extractall(dest)
    os.remove(src)

def resize(path, min_size, preprocess=lambda image, file: image, load=imageio.imread):
    min_size = np.asarray(min_size)

    tempfile_marker = "__resize-images-temp-marker"
    def to_tempfile(file):
        file, extension = os.path.splitext(file)
        return file + tempfile_marker + extension
    def from_tempfile(file):
        file, extension = os.path.splitext(file)
        assert file.endswith(tempfile_marker)
        file = file[:-len(tempfile_marker)]
        return file + extension
    def is_tempfile(file):
        file, extension = os.path.splitext(file)
        return file.endswith(tempfile_marker)

    tasks = []
    for subpath, dirs, files in os.walk(path):
        for file in files:
            if tfcv.dataset.util.is_image_file(file):
                file = os.path.join(subpath, file)
                if is_tempfile(file):
                    tempfile = file
                    file = from_tempfile(file)
                    if os.path.isfile(file):
                        print(f"Found temp-file {tempfile}, removing")
                        os.remove(tempfile)
                    else:
                        print(f"Found temp-file {tempfile}, renaming to {os.path.basename(file)}")
                        os.rename(tempfile, file)
                else:
                    tasks.append(file)
    print(f"Found {len(tasks)} images in {path} that will be resized")

    stream = iter(tasks)
    stream = pl.sync(stream) # Concurrent processing starts after this point

    # Load frame
    def load2(file):
        try:
            image = load(file)
        except ValueError as e:
            print(f"Failed to resize image file {file}")
            raise e
        if np.any(np.asarray(image.shape[:2]) != min_size) or not file.endswith(".jpg"):
            return [(image, file)]
        else:
            return []
    stream = pl.map(load2, stream)
    stream = pl.flatten(stream)
    stream = pl.queued(stream, workers=4, maxsize=4)

    # Convert frame
    @pl.unpack
    def convert(image, file):
        image = preprocess(image, file)

        shape = np.asarray(image.shape[:2])
        factor = np.amax(min_size.astype("float") / shape)
        shape = (shape * factor).astype("int")

        image = tfcv.image.resize_to(shape)((image, "color"))
        return image, file
    stream = pl.map(convert, stream)
    stream = pl.queued(stream, workers=12, maxsize=12)

    # Save frame
    @pl.unpack
    def save(image, file_png):
        file_jpg = ".".join(file_png.split(".")[:-1]) + ".jpg"
        tempfile_jpg = to_tempfile(file_jpg)
        imageio.imwrite(tempfile_jpg, image)
        os.remove(file_png)
        os.rename(tempfile_jpg, file_jpg)
    stream = pl.map(save, stream)
    stream = pl.queued(stream, workers=4, maxsize=4)

    for _ in tqdm.tqdm(stream, total=len(tasks)):
        pass

class RunException(BaseException):
    def __init__(self, message, code):
        self.message = message
        self.code = code

def run(command):
    print("> " + command)
    returncode = os.system(f"bash -c '{command}'")
    if returncode != 0:
        raise RunException("Failed to run " + command + ". Got return code " + str(returncode), returncode)

def pano_to_pinhole(image, shape, intr, theta, phi):
    def xyz2lonlat(xyz):
        atan2 = np.arctan2
        asin = np.arcsin

        norm = np.linalg.norm(xyz, axis=-1, keepdims=True)
        xyz_norm = xyz / norm
        x = xyz_norm[..., 0:1]
        y = xyz_norm[..., 1:2]
        z = xyz_norm[..., 2:]

        lon = atan2(x, z)
        lat = asin(y)
        lst = [lon, lat]

        out = np.concatenate(lst, axis=-1)
        return out

    def lonlat2XY(lonlat, shape):
        X = (lonlat[..., 0:1] / (2 * np.pi) + 0.5) * (shape[1] - 1)
        Y = (lonlat[..., 1:] / (np.pi) + 0.5) * (shape[0] - 1)
        lst = [X, Y]
        out = np.concatenate(lst, axis=-1)

        return out

    K = np.asarray(intr)
    K_inv = np.linalg.inv(K)

    x = np.arange(shape[1])
    y = np.arange(shape[0])
    x, y = np.meshgrid(x, y)
    z = np.ones_like(x)
    xyz = np.concatenate([x[..., None], y[..., None], z[..., None]], axis=-1)
    xyz = xyz @ K_inv.T

    y_axis = np.array([0.0, 1.0, 0.0], np.float32)
    x_axis = np.array([1.0, 0.0, 0.0], np.float32)
    R1, _ = cv2.Rodrigues(y_axis * theta)
    R2, _ = cv2.Rodrigues(np.dot(R1, x_axis) * phi)
    R = R2 @ R1
    xyz = xyz @ R.T
    lonlat = xyz2lonlat(xyz)
    XY = lonlat2XY(lonlat, shape=image.shape[:2]).astype(np.float32)
    pinhole = cv2.remap(image, XY[..., 0], XY[..., 1], cv2.INTER_CUBIC, borderMode=cv2.BORDER_WRAP)

    return pinhole
