import os, imageio, tfcv
import numpy as np

class Frame:
    def __init__(self, images, frame_id):
        self.images = images
        self.frame_id = frame_id

class SingleTileFrameId:
    def __init__(self, tile, files):
        self.tile = tile
        self.files = files

    def has_missing(self):
        return None in self.files

    def load(self):
        images = [imageio.imread(f) for f in self.files]

        return Frame(images, self)

class MultiTileFrameId:
    def __init__(self, sub_frame_ids):
        self.sub_frame_ids = sub_frame_ids
        self.tile = self.sub_frame_ids[0][0].tile

    def has_missing(self):
        return any([any([l2.has_missing() for l2 in l1]) for l1 in self.sub_frame_ids])

    def load(self):
        images = []
        for i in range(len(self.sub_frame_ids[0][0].files)):
            if any([any([None in l2.files for l2 in l1]) for l1 in self.sub_frame_ids]):
                # Not all surrounding tiles have this image type
                images.append(None)
            else:
                tiles_num = (len(self.sub_frame_ids), len(self.sub_frame_ids[0]))
                image = None
                for c in range(tiles_num[0]): # latitude
                    for r in range(tiles_num[1]): # longitude
                        sub_image = imageio.imread(self.sub_frame_ids[c][r].files[i])
                        sub_tile = np.asarray([c, r])
                        if image is None:
                            image = np.zeros((tiles_num[1] * sub_image.shape[1], tiles_num[0] * sub_image.shape[0], sub_image.shape[2]), dtype=sub_image.dtype)

                        min_pixel = sub_tile * np.asarray(sub_image.shape[:2])
                        max_pixel = (sub_tile + 1) * np.asarray(sub_image.shape[:2])
                        image[min_pixel[1]:max_pixel[1], min_pixel[0]:max_pixel[0]] = sub_image
                images.append(image)

        return Frame(images, self)

def load(*paths, tiles=(1, 1), allow_missing=False):
    frame_ids = {}
    for i, path in enumerate(paths):
        for z in os.listdir(path):
            z_path = os.path.join(path, z)
            if os.path.isdir(z_path):
                for x in os.listdir(z_path):
                    zx_path = os.path.join(z_path, x)
                    if os.path.isdir(zx_path):
                        for y in os.listdir(zx_path):
                            file = os.path.join(zx_path, y)
                            if tfcv.data.util.is_image_file(file):
                                tile = tuple([int(x), int(y.split(".")[0]), int(z)])
                                if not tile in frame_ids:
                                    frame_ids[tile] = SingleTileFrameId(tile, [None] * len(paths))
                                frame_ids[tile].files[i] = file

    if tiles != (1, 1):
        combined_frame_ids = []
        for frame_id in frame_ids.values():
            # Find all surrounding tiles
            sub_frame_ids = [[None] * tiles[1] for _ in range(tiles[0])]
            skip = False
            for tile_x in range(tiles[0]):
                for tile_y in range(tiles[1]):
                    tile = (tile_x + frame_id.tile[0] - tiles[0] // 2, tile_y + frame_id.tile[1] - tiles[1] // 2, frame_id.tile[2])
                    if tile in frame_ids:
                        sub_frame_ids[tile_x][tile_y] = frame_ids[tile]
                    else:
                        skip = True
                        break
                if skip:
                    break
            if not skip:
                # If all surrounding tiles could be found
                combined_frame_ids.append(MultiTileFrameId(sub_frame_ids))
        frame_ids = combined_frame_ids
    else:
        frame_ids = list(frame_ids.values())

    if not allow_missing:
        frame_ids = [frame_id for frame_id in frame_ids if not frame_id.has_missing()]
    frame_ids = sorted(frame_ids, key=lambda frame_id: frame_id.tile)
    return frame_ids
