#pragma once

#include <xti/typedefs.h>
#include <xti/exception.h>
#include <xti/opencv.h>
#include <tiledwebmaps/tileloader.h>
#include <curl_easy.h>
#include <curl_header.h>
#include <regex>
#include <thread>
#include <chrono>

namespace tiledwebmaps {

class Http : public TileLoader
{
public:
  Http(const Layout& layout, size_t retries = 10, float wait_after_error = 1.5)
    : TileLoader(layout)
    , m_retries(retries)
    , m_wait_after_error(wait_after_error)
  {
  }

  Tile load(xti::vec2s tile, size_t zoom)
  {
    std::string url = this->get_url(tile, zoom);
    std::map<std::string, std::string> header;

    LoadTileException last_ex;
    for (size_t tries = 0; tries < m_retries; ++tries)
    {
      if (tries > 0)
      {
        std::this_thread::sleep_for(std::chrono::duration<float>(m_wait_after_error));
      }
      try
      {
        // Retrieve data from url
        curl::curl_easy request;
        request.add<CURLOPT_URL>(url.c_str());
        request.add<CURLOPT_FOLLOWLOCATION>(1L);

        curl::curl_header curl_header;
        for (const auto& pair : header)
        {
          curl_header.add(pair.first + ": " + pair.second);
        }
        request.add<CURLOPT_HTTPHEADER>(curl_header.get());

        std::ostringstream header_stream, body_stream;
        curl::curl_ios<std::ostringstream> curl_header_stream(header_stream);
        curl::curl_ios<std::ostringstream> curl_body_stream(body_stream);
        request.add<CURLOPT_HEADERFUNCTION>(curl_header_stream.get_function());
        request.add<CURLOPT_HEADERDATA>(curl_header_stream.get_stream());
        request.add<CURLOPT_WRITEFUNCTION>(curl_body_stream.get_function());
        request.add<CURLOPT_WRITEDATA>(curl_body_stream.get_stream());

        request.perform();

        // Convert data to image
        std::string data = body_stream.str();
        cv::Mat data_cv(1, data.length(), xti::opencv::pixeltype<uint8_t>::get(1), data.data());
        if (data_cv.data == NULL)
        {
          last_ex = LoadTileException("Failed to download image from url " + url);
          continue;
        }
        cv::Mat image_cv = cv::imdecode(data_cv, cv::IMREAD_COLOR);
        if (image_cv.data == NULL)
        {
          last_ex = LoadTileException("Failed to decode downloaded image from url " + url + ". Received " + XTI_TO_STRING(data.length()) + " bytes: " + data);
          continue;
        }
        auto image_bgr = xti::from_opencv<uint8_t>(std::move(image_cv));
        auto image_rgb = xt::view(std::move(image_bgr), xt::all(), xt::all(), xt::range(xt::placeholders::_, xt::placeholders::_, -1));
        try
        {
          return to_tile(std::move(image_rgb));
        }
        catch (LoadTileException ex)
        {
          last_ex = LoadTileException(std::string("Downloaded invalid tile. ") + ex.what());
        }
      }
      catch (curl::curl_easy_exception ex)
      {
        last_ex = LoadTileException(std::string("Failed to download image. Reason: ") + ex.what());
      }
    }
    throw last_ex;
  }

  virtual std::string get_url(xti::vec2s tile, size_t zoom) const = 0;

private:
  size_t m_retries;
  float m_wait_after_error;
};

class XYZ : public Http
{
public:
  template <typename... TArgs>
  XYZ(std::string url, TArgs&&... args)
    : Http(std::forward<TArgs>(args)...)
    , m_url(url)
  {
  }

  std::string get_url(xti::vec2s tile, size_t zoom) const
  {
    std::string url = m_url;
    url = std::regex_replace(url, std::regex("\\{x\\}"), std::to_string(tile(0)));
    url = std::regex_replace(url, std::regex("\\{y\\}"), std::to_string(tile(1)));
    url = std::regex_replace(url, std::regex("\\{zoom\\}"), std::to_string(zoom));
    url = std::regex_replace(url, std::regex("\\{z\\}"), std::to_string(zoom));
    return url;
  }

private:
  std::string m_url;
};

class Quad : public Http
{
public:
  template <typename... TArgs>
  Quad(std::string url, TArgs&&... args)
    : Http(std::forward<TArgs>(args)...)
    , m_url(url)
  {
  }

  std::string get_url(xti::vec2s tile, size_t zoom) const
  {
    std::string quad = "";
    for (int32_t bit = zoom; bit > 0; bit--)
    {
      char digit = '0';
      auto mask = 1 << (bit - 1);
      if ((tile(0) & mask) != 0)
      {
        digit += 1;
      }
      if ((tile(1) & mask) != 0)
      {
        digit += 2;
      }
      quad += digit;
    }

    std::string url = m_url;
    url = std::regex_replace(url, std::regex("\\{quad\\}"), quad);
    url = std::regex_replace(url, std::regex("\\{q\\}"), quad);
    return url;
  }

private:
  std::string m_url;
};

class WMS : public Http
{
public:
  template <typename... TArgs>
  WMS(std::string url, TArgs&&... args)
    : Http(std::forward<TArgs>(args)...)
    , m_url(url)
  {
  }

  std::string get_url(xti::vec2s tile, size_t zoom) const
  {
    xti::vec2d tile_lower_bound = this->get_layout().tile_to_crs(tile, zoom);
    xti::vec2d tile_upper_bound = this->get_layout().tile_to_crs(tile + 1, zoom);

    std::string bbox = std::to_string(tile_lower_bound(0)) + "%2C" + std::to_string(tile_lower_bound(1)) + "%2C" + std::to_string(tile_upper_bound(0)) + "%2C" + std::to_string(tile_upper_bound(1));
    std::string size = std::to_string(this->get_layout().get_tile_shape()(0)) + "%2C" + std::to_string(this->get_layout().get_tile_shape()(1));

    std::string url = m_url;
    url = std::regex_replace(url, std::regex("\\{bbox\\}"), bbox);
    url = std::regex_replace(url, std::regex("\\{size\\}"), size);
    return url;
  }

private:
  std::string m_url;
};



} // end of ns tiledwebmaps
