from typing import Type, Any, Callable, Union, List, Optional
import numpy as np
import torch
from torch import Tensor
import torch.nn as nn
import torch.nn.functional as F
import torch_geometric
import torch_geometric.nn as gnn
from torch_geometric.data import Data, Batch


def load_param(model, model_path, map_location=None, verbose=True):
    model_state_dict = model.state_dict()
    loaded_state_dict = torch.load(model_path, map_location=map_location)
    for key in loaded_state_dict:
        if key not in model_state_dict or model_state_dict[key].shape != loaded_state_dict[key].shape:
            if verbose: print(f'load_param: ignore {key}')
            continue
        model_state_dict[key].copy_(loaded_state_dict[key])


def unbatch(src: Tensor, batch: Tensor, dim: int = 0) -> List[Tensor]:
    r"""Splits :obj:`src` according to a :obj:`batch` vector along dimension
    :obj:`dim`.

    Args:
        src (Tensor): The source tensor.
        batch (LongTensor): The batch vector
            :math:`\mathbf{b} \in {\{ 0, \ldots, B-1\}}^N`, which assigns each
            entry in :obj:`src` to a specific example. Must be ordered.
        dim (int, optional): The dimension along which to split the :obj:`src`
            tensor. (default: :obj:`0`)

    :rtype: :class:`List[Tensor]`
    """
    sizes = torch_geometric.utils.degree(batch, dtype=torch.long).tolist()
    return src.split(sizes, dim)


def conv3x3(in_planes: int, out_planes: int, stride: int = 1, groups: int = 1, dilation: int = 1) -> nn.Conv2d:
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes: int, out_planes: int, stride: int = 1) -> nn.Conv2d:
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion: int = 1

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion: int = 4

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNetEncoder(nn.Module):

    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck]],
        layers: List[int],
        num_channels: int = 3,
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., nn.Module]] = None,
    ) -> None:
        super(ResNetEncoder, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.block = block

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(num_channels, self.inplanes, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        # self.avgpool = nn.AvgPool2d(8,8,ceil_mode=True)
        # self.avgpool2 = nn.AvgPool2d(4,4,ceil_mode=True)
        # self.final_conv = nn.Sequential(nn.Conv2d(512*block.expansion+64*block.expansion+128*block.expansion, 512*block.expansion,1),
        #                                 norm_layer(512*block.expansion),
        #                                 nn.ReLU(inplace=True))

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: Type[Union[BasicBlock, Bottleneck]], planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False) -> nn.Sequential:
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    # def _forward_impl(self, x: Tensor) -> Tensor:
    #     # See note [TorchScript super()]
    #     x = self.conv1(x)
    #     x = self.bn1(x)
    #     x = self.relu(x)
    #     x = self.maxpool(x)

    #     x1 = self.layer1(x) #x = self.layer1(x) 
    #     x2 = self.layer2(x1) #x = self.layer2(x)
    #     x = self.layer3(x2) #x = self.layer3(x)
    #     x = self.layer4(x)

    #     x = self.final_conv(torch.cat([x, self.avgpool(x1), self.avgpool2(x2)], dim=1))
    #     return x
    def _forward_impl(self, x: Tensor) -> Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        return x

    def forward(self, x: Tensor) -> Tensor:
        return self._forward_impl(x)


class BiGCN_Graph(nn.Module):
    hidden_channels = 512

    def get_full_edge_index(self):
        fea_size = [(self.input_size[0]+31)//32, (self.input_size[1]+31)//32]
        self.fea_size = fea_size
        self.len_fea = fea_size[0] * fea_size[1]
        adj_matrix = torch.zeros(self.len_fea, self.len_fea, dtype=torch.long)
        s_l, t_l = [], []
        neighbor_list = []
        for i in range(fea_size[0]):
            for j in range(fea_size[1]):
                s = i * fea_size[1] + j
                s_neighbor = []
                for di in [-1,0,1]:
                    ti = i + di
                    if ti < 0 or ti >= fea_size[0]: continue
                    for dj in [-1,0,1]:
                        tj = j + dj
                        if tj < 0 or tj >= fea_size[1] or di == 0 and dj == 0:
                            continue
                        t = ti * fea_size[1] + tj
                        s_l.append(s)
                        t_l.append(t)
                        adj_matrix[s,t] = 1
                        s_neighbor.append(t)
                neighbor_list.append(s_neighbor)
        return torch.tensor([s_l, t_l], dtype=torch.long), adj_matrix, neighbor_list

    def __init__(self, n_seg_classes, input_size, fea_channels):
        super(BiGCN_Graph, self).__init__()
        self.n_seg_classes = n_seg_classes
        self.input_size = input_size

        #self.g_proj = nn.Conv2d(fea_channels, self.hidden_channels, 1)
        full_edge_index, full_adj_matrix, neighbor_list = self.get_full_edge_index()
        self.register_buffer('full_edge_index', full_edge_index)
        self.register_buffer('full_adj_matrix', full_adj_matrix)
        self.neighbor_list = neighbor_list
        self.full_edge_set = set(map(lambda x: tuple(x), full_edge_index.transpose(0,1).tolist()))

        self.node_classify = gnn.Sequential(input_args='x, edge_index', modules=[
            (gnn.GCNConv(fea_channels, self.hidden_channels), 'x, edge_index -> x1'),
            gnn.BatchNorm(self.hidden_channels),
            nn.ReLU(inplace=True),
            nn.Dropout(p=0.2),
            (gnn.GCNConv(self.hidden_channels, self.hidden_channels), 'x1, edge_index -> x2'),
            gnn.BatchNorm(self.hidden_channels),
            nn.ReLU(inplace=True),
            nn.Dropout(p=0.1),
            (gnn.GCNConv(self.hidden_channels, n_seg_classes), 'x2, edge_index -> x3'),
        ])

    def forward(self, fea):
        #fea = self.g_proj(fea)
        g_datas = []
        for i in range(fea.shape[0]):
            g_datas.append(Data(x=fea[i].permute(1,2,0).flatten(0,1), edge_index=self.full_edge_index))
        data = Batch.from_data_list(g_datas)

        cls_logit = torch.stack(unbatch(self.node_classify(data.x, data.edge_index), data.batch), dim=0) #[batch, n_nodes, n_seg_classes]
        cls_logit = cls_logit.view(fea.shape[0], self.fea_size[0], self.fea_size[1], self.n_seg_classes).permute(0,3,1,2) #[batch, n_seg_classes, fea_h, fea_w]

        return cls_logit


class BiGCN_Nolink(nn.Module):

    def __init__(self, n_channels, n_cls_classes, n_seg_classes, input_size):
        super(BiGCN_Nolink, self).__init__()
        self.n_channels = n_channels
        self.n_cls_classes = n_cls_classes
        self.n_seg_classes = n_seg_classes

        self.resnet50 = ResNetEncoder(Bottleneck, [3, 4, 6, 3], n_channels)

        #self.cls_branch = nn.Sequential(nn.Conv2d(512 * self.resnet50.block.expansion, n_cls_classes, 1), nn.AdaptiveAvgPool2d(1), nn.Flatten(1))
        self.cls_branch = nn.Sequential(nn.AdaptiveAvgPool2d(1), nn.Flatten(1), nn.Linear(512 * self.resnet50.block.expansion, n_cls_classes))

        if isinstance(input_size, (int, float)):
            input_size = [int(input_size), int(input_size)]
        self.graph_branch = BiGCN_Graph(n_seg_classes, input_size, 512 * self.resnet50.block.expansion)

    def forward(self, x, mode=None):
        fea = self.resnet50(x)
        if mode == 'cls':
            return self.cls_branch(fea)
        elif mode == 'seg':
            return self.graph_branch(fea)
        else:
            cls_logit = self.cls_branch(fea)
            seg_cls_logit = self.graph_branch(fea)
            return cls_logit, seg_cls_logit
