from typing import Type, Any, Callable, Union, List, Optional
import numpy as np
import torch
from torch import Tensor
import torch.nn as nn
import torch.nn.functional as F
import torch_geometric
import torch_geometric.nn as gnn
from torch_geometric.data import Data, Batch
from .backbones import load_backbone


def unbatch(src: Tensor, batch: Tensor, dim: int = 0) -> List[Tensor]:
    r"""Splits :obj:`src` according to a :obj:`batch` vector along dimension
    :obj:`dim`.

    Args:
        src (Tensor): The source tensor.
        batch (LongTensor): The batch vector
            :math:`\mathbf{b} \in {\{ 0, \ldots, B-1\}}^N`, which assigns each
            entry in :obj:`src` to a specific example. Must be ordered.
        dim (int, optional): The dimension along which to split the :obj:`src`
            tensor. (default: :obj:`0`)

    :rtype: :class:`List[Tensor]`
    """
    sizes = torch_geometric.utils.degree(batch, dtype=torch.long).tolist()
    return src.split(sizes, dim)


class BiGCN_Graph(nn.Module):
    hidden_channels = 512

    def get_full_edge_index(self):
        fea_size = [(self.input_size[0]+31)//32, (self.input_size[1]+31)//32]
        self.fea_size = fea_size
        self.len_fea = fea_size[0] * fea_size[1]
        adj_matrix = torch.zeros(self.len_fea, self.len_fea, dtype=torch.long)
        s_l, t_l = [], []
        neighbor_list = []
        for i in range(fea_size[0]):
            for j in range(fea_size[1]):
                s = i * fea_size[1] + j
                s_neighbor = []
                for di in [-1,0,1]:
                    ti = i + di
                    if ti < 0 or ti >= fea_size[0]: continue
                    for dj in [-1,0,1]:
                        tj = j + dj
                        if tj < 0 or tj >= fea_size[1] or di == 0 and dj == 0:
                            continue
                        t = ti * fea_size[1] + tj
                        s_l.append(s)
                        t_l.append(t)
                        adj_matrix[s,t] = 1
                        s_neighbor.append(t)
                neighbor_list.append(s_neighbor)
        return torch.tensor([s_l, t_l], dtype=torch.long), adj_matrix, neighbor_list

    def __init__(self, n_seg_classes, input_size, fea_channels):
        super(BiGCN_Graph, self).__init__()
        self.n_seg_classes = n_seg_classes
        self.input_size = input_size

        full_edge_index, full_adj_matrix, neighbor_list = self.get_full_edge_index()
        self.register_buffer('full_edge_index', full_edge_index)
        self.register_buffer('full_adj_matrix', full_adj_matrix)
        self.neighbor_list = neighbor_list
        self.full_edge_set = set(map(lambda x: tuple(x), full_edge_index.transpose(0,1).tolist()))

        self.node_classify = gnn.Sequential(input_args='x, edge_index', modules=[
            (gnn.GCNConv(fea_channels, self.hidden_channels), 'x, edge_index -> x1'),
            gnn.BatchNorm(self.hidden_channels),
            nn.ReLU(inplace=True),
            nn.Dropout(p=0.2),
            (gnn.GCNConv(self.hidden_channels, self.hidden_channels), 'x1, edge_index -> x2'),
            gnn.BatchNorm(self.hidden_channels),
            nn.ReLU(inplace=True),
            nn.Dropout(p=0.1),
            (gnn.GCNConv(self.hidden_channels, n_seg_classes), 'x2, edge_index -> x3'),
        ])

    def forward(self, fea):
        g_datas = []
        for i in range(fea.shape[0]):
            g_datas.append(Data(x=fea[i].permute(1,2,0).flatten(0,1), edge_index=self.full_edge_index))
        data = Batch.from_data_list(g_datas)

        cls_logit = torch.stack(unbatch(self.node_classify(data.x, data.edge_index), data.batch), dim=0) #[batch, n_nodes, n_seg_classes]
        cls_logit = cls_logit.view(fea.shape[0], self.fea_size[0], self.fea_size[1], self.n_seg_classes).permute(0,3,1,2) #[batch, n_seg_classes, fea_h, fea_w]

        return cls_logit


class BiGCN_Nolink(nn.Module):

    def __init__(self, n_channels, n_cls_classes, n_seg_classes, input_size, backbone='resnet50'):
        super(BiGCN_Nolink, self).__init__()
        self.n_channels = n_channels
        self.n_cls_classes = n_cls_classes
        self.n_seg_classes = n_seg_classes

        self.backbone, backbone_out_channel = load_backbone(backbone, n_channels)

        self.cls_branch = nn.Sequential(nn.AdaptiveAvgPool2d(1), nn.Flatten(1), nn.Linear(backbone_out_channel, n_cls_classes))

        if isinstance(input_size, (int, float)):
            input_size = [int(input_size), int(input_size)]
        self.graph_branch = BiGCN_Graph(n_seg_classes, input_size, backbone_out_channel)

    def forward(self, x, mode=None):
        fea = self.backbone(x)
        if mode == 'cls':
            return self.cls_branch(fea)
        elif mode == 'seg':
            return self.graph_branch(fea)
        else:
            cls_logit = self.cls_branch(fea)
            seg_cls_logit = self.graph_branch(fea)
            return cls_logit, seg_cls_logit
