import numpy as np
import torch
import torch.nn as nn
from copy import deepcopy
from torch.nn.parallel import DistributedDataParallel

# A wrapper model for Classifier-free guidance **SAMPLING** only
# https://arxiv.org/abs/2207.12598
class ClassifierFreeSampleModel(nn.Module):

    def __init__(self, model):
        super().__init__()
        self.model = model  # model is the actual model to run
        self.module = model.module if isinstance(model, DistributedDataParallel) else model
        
        assert self.module.cond_mask_prob > 0, 'Cannot run a guided diffusion on a model that has not been trained with no conditions'

        # # pointers to inner model
        self.rot2xyz = self.module.rot2xyz
        self.translation = self.module.translation
        self.njoints = self.module.njoints
        self.nfeats = self.module.nfeats
        self.data_rep = self.module.data_rep
        self.cond_mode = self.module.cond_mode
        self.encode_text = self.module.encode_text

    def forward(self, x, timesteps, y=None):
        cond_mode = self.cond_mode
        assert cond_mode in ['text', 'action']
        y_uncond = deepcopy(y)
        y_uncond['uncond'] = True
        out = self.model(x, timesteps, y)
        out_uncond = self.model(x, timesteps, y_uncond)
        return out_uncond + (y['scale'].view(-1, 1, 1, 1) * (out - out_uncond))

