import sys

import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid
import shutil

from llava.model import *
from llava.constants import DEFAULT_IMAGE_PATCH_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle

from llava.utils import disable_torch_init
from llava.mm_utils import  get_model_name_from_path, tokenizer_multi_modal_token
from torch.utils.data import Dataset, DataLoader

from PIL import Image
import math

from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig, BitsAndBytesConfig

def load_pretrained_model(model_path, model_base, model_name, load_8bit=False, load_4bit=False, device_map="auto", device="cuda"):
    kwargs = {"device_map": device_map}

    if load_8bit:
        kwargs['load_in_8bit'] = True
    elif load_4bit:
        kwargs['load_in_4bit'] = True
        kwargs['quantization_config'] = BitsAndBytesConfig(
            load_in_4bit=True,
            bnb_4bit_compute_dtype=torch.float16,
            bnb_4bit_use_double_quant=True,
            bnb_4bit_quant_type='nf4'
        )
    else:
        kwargs['torch_dtype'] = torch.float16

    # Load LLaVA model
    if model_base is not None:
        print('Loading LLaVA from base model...')
        if 'mpt' in model_name.lower():
            if not os.path.isfile(os.path.join(model_path, 'configuration_mpt.py')):
                shutil.copyfile(os.path.join(model_base, 'configuration_mpt.py'), os.path.join(model_path, 'configuration_mpt.py'))
            tokenizer = AutoTokenizer.from_pretrained(model_base, use_fast=True)
            cfg_pretrained = AutoConfig.from_pretrained(model_path, trust_remote_code=True)
            model = LlavaMPTForCausalLM.from_pretrained(model_base, low_cpu_mem_usage=True, config=cfg_pretrained, **kwargs)
        else:
            tokenizer = AutoTokenizer.from_pretrained(model_base, use_fast=False)
            cfg_pretrained = AutoConfig.from_pretrained(model_path)
            model = LlavaLlamaForCausalLM.from_pretrained(model_base, low_cpu_mem_usage=True, config=cfg_pretrained, **kwargs)

        mm_projector_weights = torch.load(os.path.join(model_path, 'mm_projector.bin'), map_location='cpu')
        mm_projector_weights = {k: v.to(torch.float16) for k, v in mm_projector_weights.items()}
        model.load_state_dict(mm_projector_weights, strict=False)
    else:
        if 'mpt' in model_name.lower():
            tokenizer = AutoTokenizer.from_pretrained(model_path, use_fast=True)
            model = LlavaMPTForCausalLM.from_pretrained(model_path, low_cpu_mem_usage=True, **kwargs)
        else:
            tokenizer = AutoTokenizer.from_pretrained(model_path, use_fast=False)
            model = LlavaLlamaForCausalLM.from_pretrained(model_path, low_cpu_mem_usage=True, **kwargs)

    image_processor = None
    chart_processor = None

    mm_use_im_start_end = getattr(model.config, "mm_use_im_start_end", False)
    mm_use_im_patch_token = getattr(model.config, "mm_use_im_patch_token", True)
    if mm_use_im_patch_token:
        tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
    if mm_use_im_start_end:
        tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)
    model.resize_token_embeddings(len(tokenizer))

    
    vision_tower = model.get_modal_tower(modal="image")
    if not vision_tower.is_loaded:
        vision_tower.load_model()
    vision_tower.to(device=device, dtype=torch.float16)
    image_processor = vision_tower.image_processor
    
    chart_tower = model.get_modal_tower(modal="chart")
    if not chart_tower.is_loaded:
        chart_tower.load_model()
    chart_tower.to(device=device, dtype=torch.float16)
    chart_processor = chart_tower.chart_processor

    if hasattr(model.config, "max_sequence_length"):
        context_len = model.config.max_sequence_length
    else:
        context_len = 2048

    return tokenizer, model, image_processor, chart_processor, context_len

def split_list(lst, n):
    """Split a list into n (roughly) equal-sized chunks"""
    chunk_size = math.ceil(len(lst) / n)  # integer division
    return [lst[i:i+chunk_size] for i in range(0, len(lst), chunk_size)]


def get_chunk(lst, n, k):
    chunks = split_list(lst, n)
    return chunks[k]


# Custom dataset class
class CustomDataset(Dataset):
    def __init__(self, questions,image_folder, chart_folder, tokenizer, image_processor,chart_processor, model_config):
        self.questions = questions
        self.image_folder = image_folder
        self.chart_folder = chart_folder
        self.tokenizer = tokenizer
        self.image_processor = image_processor
        self.chart_processor = chart_processor
        self.model_config = model_config

    def __getitem__(self, index):
        line = self.questions[index]
        
        if "image" in line: assert self.image_folder!=None and self.image_processor!=None
        if "chart" in line: assert self.chart_folder!=None and self.chart_processor!=None
        
        if "chart" in line :
            chart_path = os.path.join(self.chart_folder,line['chart'])
            chart = Image.open(chart_path).convert('RGB')
            chart = self.chart_processor(chart, return_tensors="pt").pixel_values[0]
        elif self.chart_processor!=None :
            crop_size = self.chart_processor.image_processor.size
            chart = torch.zeros(3, crop_size['height'], crop_size['width'])
        if "image" in line :
            image_path = os.path.join(self.image_folder,line['image'])
            image = Image.open(image_path).convert('RGB')
            image = self.image_processor.preprocess(image, return_tensors="pt").pixel_values[0]
        elif self.image_processor!=None:
            crop_size = self.image_processor.crop_size
            image = torch.zeros(3, crop_size['height'], crop_size['width'])

        qs = line["conversations"][0]["value"]

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()
        
        input_ids = tokenizer_multi_modal_token(prompt,self.tokenizer)

        return input_ids, chart, image

    def __len__(self):
        return len(self.questions)


# DataLoader
def create_data_loader(questions, image_folder, chart_folder, tokenizer, image_processor, chart_processor, model_config, batch_size=1, num_workers=4):
    assert batch_size == 1, "batch_size must be 1"
    dataset = CustomDataset(questions,image_folder, chart_folder, tokenizer, image_processor,chart_processor, model_config)
    data_loader = DataLoader(dataset, batch_size=batch_size, num_workers=num_workers, shuffle=False)
    return data_loader


def eval_model(args):
    # Model
    disable_torch_init()
    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)
    tokenizer, model, image_processor, chart_processor, context_len = load_pretrained_model(model_path, args.model_base, model_name)

    questions = json.load(open(os.path.expanduser(args.question_file), "r"))
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    answers_file = os.path.expanduser(args.answers_file)
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")

    data_loader = create_data_loader(questions, args.image_folder,args.chart_folder, tokenizer, image_processor,chart_processor, model.config)

    for (input_ids, chart_tensor,image_tensor), line in tqdm(zip(data_loader, questions), total=len(questions)):
        idx = line["id"]
        cur_prompt = line["conversations"][0]["value"]
        gold =line["conversations"][1]["value"]

        stop_str = conv_templates[args.conv_mode].sep if conv_templates[args.conv_mode].sep_style != SeparatorStyle.TWO else conv_templates[args.conv_mode].sep2
        input_ids = input_ids.to(device='cuda', non_blocking=True)

        with torch.inference_mode():
            output_ids = model.generate(
                input_ids,
                charts=chart_tensor.to(dtype=torch.float16, device='cuda', non_blocking=True),
                images = image_tensor.to(dtype=torch.float16, device='cuda', non_blocking=True),
                do_sample=True if args.temperature > 0 else False,
                temperature=args.temperature,
                top_p=args.top_p,
                num_beams=args.num_beams,
                max_new_tokens=1024,
                use_cache=True)

        input_token_len = input_ids.shape[1]
        n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
        if n_diff_input_output > 0:
            print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
        outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
        outputs = outputs.strip()
        if outputs.endswith(stop_str):
            outputs = outputs[:-len(stop_str)]
        outputs = outputs.strip()

        ans_id = shortuuid.uuid()
        ans_file.write(json.dumps({"id": idx,
                                   "prompt": cur_prompt,
                                   "pred": outputs,
                                    "gold": gold,
                                   "answer_id": ans_id,
                                   "model_id": model_name,
                                   "metadata": {}}) + "\n")
        # ans_file.flush()
    ans_file.close()

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model-path", type=str, default="facebook/opt-350m")
    parser.add_argument("--model-base", type=str, default=None)
    parser.add_argument("--image-folder", type=str, default="")
    parser.add_argument("--chart-folder", type=str, default="")
    parser.add_argument("--question-file", type=str, default="tables/question.jsonl")
    parser.add_argument("--answers-file", type=str, default="answer.jsonl")
    parser.add_argument("--conv-mode", type=str, default="vicuna_v1")
    parser.add_argument("--num-chunks", type=int, default=1)
    parser.add_argument("--chunk-idx", type=int, default=0)
    parser.add_argument("--temperature", type=float, default=0.0)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=1)
    args = parser.parse_args()

    eval_model(args)
