#    Copyright 2023 Haotian Liu
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from abc import ABC, abstractmethod

import torch
import torch.nn as nn

from llava.model.multimodal_encoder.donut_unichart_encoder import DonutEncoder

from .multimodal_encoder.builder import build_vision_tower,build_modal_tower
from .multimodal_projector.builder import build_modal_projector
from transformers import VisionEncoderDecoderModel
from llava.constants import IGNORE_INDEX, IMAGE_TOKEN_INDEX, CHART_TOKEN_INDEX,DEFAULT_IMAGE_PATCH_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN,DEFAULT_IMAGE_TOKEN, DEFAULT_CHART_TOKEN

class LlavaMetaModel:

    
    def __init__(self, config):
        super(LlavaMetaModel, self).__init__(config)

        # llava로 모델 로딩 했을 때, multimodal encoder, projection layer loading하는 부분
        if hasattr(config, "mm_vision_tower"):
            # raise NotImplementedError
            # self.vision_tower = build_vision_tower(config, delay_load=True)
            # dubug
            # pass
            self.vision_tower = build_modal_tower(config,modal="image",delay_load=True)
            self.mm_projector = build_modal_projector(config.mm_projector_type, self.vision_tower.hidden_size, config.hidden_size)
        if hasattr(config, "mm_chart_tower"):
            self.chart_tower = build_modal_tower(config,modal="chart",delay_load=False)
            self.chart_mm_projector = build_modal_projector(config.mm_chart_projector_type, self.chart_tower.hidden_size, config.hidden_size)

    def get_modal_tower(self,modal="image"):
        if modal=="image":
            vision_tower = getattr(self, 'vision_tower', None)
            if type(vision_tower) is list:
                vision_tower = vision_tower[0]
            return vision_tower
        elif modal=="chart":
            chart_tower = getattr(self, 'chart_tower', None)
            if type(chart_tower) is list:
                chart_tower = chart_tower[0]
            return chart_tower
    
    def get_modal_projector(self,modal="image"):
        if modal=="image":
            return self.mm_projector
        elif modal=="chart":
            return self.chart_mm_projector
        else:
            raise NotImplementedError

    def initialize_modal_modules(self, model_args, modal="image",fsdp=None,do_print=False):
        
        def get_w(weights, keyword):
            return {k.split(keyword + '.')[1]: v for k, v in weights.items() if keyword in k}
        
        # 이미 만들어진 Modality Encoder, Projection Layer가 없다면 초기화 
        if modal=="image":
            vision_tower = model_args.vision_tower
            mm_vision_select_layer = model_args.mm_vision_select_layer
            mm_vision_select_feature = model_args.mm_vision_select_feature
            pretrain_mm_mlp_adapter = model_args.pretrain_mm_mlp_adapter

            self.config.mm_vision_tower = vision_tower


            if self.get_modal_tower(modal="image") is None:
                if do_print: print(f"Build Vision Tower from {model_args.vision_tower}")
                vision_tower = build_modal_tower(model_args,modal="image")

                if fsdp is not None and len(fsdp) > 0:
                    self.vision_tower = [vision_tower]
                else:
                    self.vision_tower = vision_tower
            else:
                if fsdp is not None and len(fsdp) > 0:
                    vision_tower = self.vision_tower[0]
                else:
                    vision_tower = self.vision_tower
                vision_tower.load_model()

            self.config.use_mm_proj = True
            self.config.mm_projector_type = getattr(model_args, 'mm_projector_type', 'linear')
            self.config.mm_hidden_size = vision_tower.hidden_size
            self.config.mm_vision_select_layer = mm_vision_select_layer
            self.config.mm_vision_select_feature = mm_vision_select_feature

            if getattr(self, 'mm_projector', None) is None:
                print("New intialize projection layer of Vision")
                self.mm_projector = build_modal_projector(self.config.mm_projector_type,self.config.mm_hidden_size,self.config.hidden_size)

            if pretrain_mm_mlp_adapter is not None:
                if do_print : print(f"Loading image projection layer weight from {model_args.pretrain_mm_mlp_adapter}")
                mm_projector_weights = torch.load(pretrain_mm_mlp_adapter, map_location='cpu')
                

                self.mm_projector.load_state_dict(get_w(mm_projector_weights, 'mm_projector'))
                
                # debug code
                # print(f"Temprory debugging / mm_projector freeze")
                # for p in self.mm_projector.parameters():
                #     p.requires_grad = False  
                
        elif modal=="chart":
            self.config.mm_chart_tower = model_args.chart_tower
            self.config.mm_chart_select_layer = model_args.mm_chart_select_layer
            self.config.mm_chart_projector_type = getattr(model_args, 'mm_chart_projector_type', 'linear')


            # print(f"Loading chart tower from {model_args.chart_tower}")
            if do_print: print(f"Build Chart Tower from {model_args.chart_tower}")
            self.chart_tower = build_modal_tower(model_args,modal="chart")
            # self.chart_tower = DonutEncoder(model_args.chart_tower,model_args)
            
            self.chart_mm_projector = build_modal_projector(model_args.mm_chart_projector_type,self.chart_tower.config.hidden_size,self.config.hidden_size)
            if model_args.pretrain_chart_mm_mlp_adapter is not None:
                if do_print : print(f"Loading chart projection layer weight from {model_args.pretrain_chart_mm_mlp_adapter}")
                chart_mm_projector_weights = torch.load(model_args.pretrain_chart_mm_mlp_adapter, map_location='cpu')
                self.chart_mm_projector.load_state_dict(get_w(chart_mm_projector_weights, 'mm_projector'))

                # debug code
                # print(f"Temprory debugging / chart_mm_projector freeze")
                # for p in self.chart_mm_projector.parameters():
                #     p.requires_grad = False  


class LlavaMetaForCausalLM(ABC):

    @abstractmethod
    def get_model(self):
        pass

    def get_vision_tower(self):
        return self.get_model().get_modal_tower("image")   
    def get_modal_tower(self,modal="image"):
        return self.get_model().get_modal_tower(modal)   

    def encode_modality(self, images,modal="image"):
        image_features = self.get_model().get_modal_tower(modal)(images)
        image_features = self.get_model().get_modal_projector(modal)(image_features)
        return image_features

    def prepare_inputs_labels_for_multimodal(
        self, input_ids, attention_mask, past_key_values, labels, modal=None
    ):
        """
            !! Modality 추가시 구현 필요 부분 !!
        """
        #TODO Multimodal 부분 추가
        # Vision Tower로 Image Encoding
        # - Input - <s> + Image embedding + caption token seq embs
        # label -  -100….. + caption token seq embs
        # vision_tower = self.get_modal_tower(modal=modal)
        #     if vision_tower is None or modal is None or input_ids.shape[1] == 1:
        #         if past_key_values is not None and vision_tower is not None and images is not None and input_ids.shape[1] == 1:
        #             attention_mask = torch.ones((attention_mask.shape[0], past_key_values[-1][-1].shape[-2] + 1), dtype=attention_mask.dtype, device=attention_mask.device)
        #         return input_ids, attention_mask, past_key_values, None, labels

        #     if type(images) is list or images.ndim == 5:
        #         concat_images = torch.cat([image for image in images], dim=0)
        #         image_features = self.encode_modality(concat_images)
        #         split_sizes = [image.shape[0] for image in images]
        #         image_features = torch.split(image_features, split_sizes, dim=0)
        #         image_features = [x.flatten(0, 1) for x in image_features]
        #     else:
        #         image_features = self.encode_modality(images)
        
        # def _encode_modalities(modal_key,modal_val):
        
        #     modal_tower = self.get_modal_tower(modal=modal_key)
        #     if modal_tower is None or modal_val is None or input_ids.shape[1] == 1:
        #         if past_key_values is not None and modal_tower is not None and modal_val is not None and input_ids.shape[1] == 1:
        #             attention_mask = torch.ones((attention_mask.shape[0], past_key_values[-1][-1].shape[-2] + 1), dtype=attention_mask.dtype, device=attention_mask.device)
        #         return input_ids, attention_mask, past_key_values, None, labels

        #     if type(modal_val) is list or modal_val.ndim == 5:
        #         concat_modalities = torch.cat([image for image in modal_val], dim=0)
        #         modal_features = self.encode_modality(concat_modalities,modal=modal_key)
        #         split_sizes = [image.shape[0] for image in modal_val]
        #         modal_features = torch.split(modal_features, split_sizes, dim=0)
        #         modal_features = [x.flatten(0, 1) for x in modal_features]
        #     else:
        #         modal_features = self.encode_modality(modal_val,modal=modal_key)
        #     return modal_features
        
        """
            1. modality 별 encoder를 활용하여 modality feature들을 생성
        """
        multi_modal_features={}
        for modal_key, modal_val_batch in modal.items():
            modal_tower = self.get_modal_tower(modal=modal_key)
            if modal_tower is None or modal_val_batch is None or input_ids.shape[1] == 1:
                # raise NotImplementedError
                if past_key_values is not None and modal_tower is not None and modal_val_batch is not None and input_ids.shape[1] == 1:
                    attention_mask = torch.ones((attention_mask.shape[0], past_key_values[-1][-1].shape[-2] + 1), dtype=attention_mask.dtype, device=attention_mask.device)
                return input_ids, attention_mask, past_key_values, None, labels

            if type(modal_val_batch) is list or modal_val_batch.ndim == 5:
                raise NotImplementedError
                concat_modalities = torch.cat([image for image in modal_val_batch], dim=0)
                modal_features = self.encode_modality(concat_modalities,modal=modal_key)
                split_sizes = [image.shape[0] for image in modal_val_batch]
                modal_features = torch.split(modal_features, split_sizes, dim=0)
                modal_features = [x.flatten(0, 1) for x in modal_features]
            else:
                modal_features=[]
                for modal_val in modal_val_batch:
                    modal_feature=self.encode_modality(modal_val.unsqueeze(dim=0),modal=modal_key)
                    modal_features.append(modal_feature.squeeze())
                    # if torch.all(modal_val==IGNORE_INDEX).item():
                    #     modal_features.append(IGNORE_INDEX)
                    # else:
                    #     modal_feature=self.encode_modality(modal_val.unsqueeze(dim=0),modal=modal_key)
                    #     modal_features.append(modal_feature.squeeze())
            # modal_features
            multi_modal_features[modal_key]=modal_features

        """
            2. batch의 example 별로 modality token 부분에 modality embedding을 끼워 넣음
        """
        new_input_embeds = []
        new_labels = [] if labels is not None else None
        # cur_modal_idx = 0
        for batch_idx, cur_input_ids in enumerate(input_ids):
            image_token_exist=(cur_input_ids == IMAGE_TOKEN_INDEX).sum()!=0
            chart_token_exist=(cur_input_ids == CHART_TOKEN_INDEX).sum()!=0
            if (cur_input_ids == IMAGE_TOKEN_INDEX).sum() == 0 and (cur_input_ids == CHART_TOKEN_INDEX).sum()==0:
                # raise NotImplementedError
                # multimodal LLM, but the current sample is not multimodal
                # FIXME: this is a hacky fix, for deepspeed zero3 to work
                # half_len = cur_input_ids.shape[0] // 2
                # cur_image_features = image_features[cur_image_idx]
                cur_input_embeds=self.get_model().embed_tokens(cur_input_ids)
                cur_image_features = multi_modal_features["image"][batch_idx]
                cur_chart_features= multi_modal_features["chart"][batch_idx]
                
                if not image_token_exist and "image" in multi_modal_features:
                    cur_image_features = multi_modal_features["image"][batch_idx]
                    cur_input_embeds=torch.cat([cur_input_embeds,cur_image_features[0:0]],dim=0)
            
                if not chart_token_exist and "chart" in multi_modal_features:
                    cur_chart_features = multi_modal_features["chart"][batch_idx]
                    cur_input_embeds=torch.cat([cur_input_embeds,cur_chart_features[0:0]],dim=0)
                # cur_input_embeds_1 = self.get_model().embed_tokens(cur_input_ids[:half_len])
                # cur_input_embeds_2 = self.get_model().embed_tokens(cur_input_ids[half_len:])
                # cur_input_embeds = torch.cat([cur_input_embeds_1, cur_image_features[0:0],cur_chart_features[0:0], cur_input_embeds_2], dim=0)
                
                cur_input_embeds = torch.cat([cur_input_embeds, cur_image_features[0:0],cur_chart_features[0:0]], dim=0)
                new_input_embeds.append(cur_input_embeds)
                if labels is not None:
                    new_labels.append(labels[batch_idx])
                # cur_image_idx += 1
                continue
            
            image_token_indices = torch.where(cur_input_ids == IMAGE_TOKEN_INDEX)[0].tolist()
            chart_token_indices = torch.where(cur_input_ids == CHART_TOKEN_INDEX)[0].tolist()
            
            multi_modal_token_indices=image_token_indices+chart_token_indices
            multi_modal_token_indices=sorted(multi_modal_token_indices)
            
            cur_new_input_embeds = []
            if labels is not None:
                cur_labels = labels[batch_idx]
                cur_new_labels = []
                assert cur_labels.shape == cur_input_ids.shape
                
            """
                2.1 cur_new_input_embeds에 modality token부분은 encoded modality로 replace
                    cur_new_input_embeds = 마지막 modality token 까지 emb
            """
            offset = 0 
            while len(multi_modal_token_indices)>0:
                modal_id = cur_input_ids[multi_modal_token_indices[0]-offset]
                modal_name=""
                if modal_id==IMAGE_TOKEN_INDEX:
                    modal_name="image"
                elif modal_id ==CHART_TOKEN_INDEX:
                    modal_name="chart"
                
                cur_modal_features = multi_modal_features[modal_name][batch_idx]
                modal_token_start = multi_modal_token_indices[0]-offset
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    raise NotImplementedError
                    # cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[:image_token_start-1]).detach())
                    # cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[image_token_start-1:image_token_start]))
                    # cur_new_input_embeds.append(cur_modal_features)
                    # cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[image_token_start+1:image_token_start+2]))
                    # if labels is not None:
                    #     cur_new_labels.append(cur_labels[:image_token_start])
                    #     cur_new_labels.append(torch.full((cur_modal_features.shape[0],), IGNORE_INDEX, device=labels.device, dtype=labels.dtype))
                    #     cur_new_labels.append(cur_labels[image_token_start:image_token_start+1])
                    #     cur_labels = cur_labels[image_token_start+2:]
                else: 
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[:modal_token_start]))
                    cur_new_input_embeds.append(cur_modal_features)
                    if labels is not None:
                        cur_new_labels.append(cur_labels[:modal_token_start])
                        cur_new_labels.append(torch.full((cur_modal_features.shape[0],), IGNORE_INDEX, device=labels.device, dtype=labels.dtype)) # modality 부분 IGNORE_INDEX로 채움
                        cur_labels = cur_labels[modal_token_start+1:]
  
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    raise NotImplementedError
                    cur_input_ids = cur_input_ids[modal_token_start+2:]
                else:
                    offset=modal_token_start+1
                    cur_input_ids = cur_input_ids[offset:]
                    
                multi_modal_token_indices.pop(0)
                # image_token_indices = torch.where(cur_input_ids == IMAGE_TOKEN_INDEX)[0]
            """
                2.2 cur_new_input_embeds += 마지막 modal_token_start 이후의 input emb
            """   
            if cur_input_ids.numel() > 0:
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    raise NotImplementedError
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids).detach())
                else:
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids))
                if labels is not None:
                    cur_new_labels.append(cur_labels)
            cur_new_input_embeds = [x.to(device=self.device) for x in cur_new_input_embeds]
            cur_new_input_embeds = torch.cat(cur_new_input_embeds, dim=0)
            
            """
                2.3 multimodal token(<chart>, <image>)이 없는 경우 처리
            """
            if not image_token_exist and "image" in multi_modal_features:
                cur_image_features = multi_modal_features["image"][batch_idx]
                cur_new_input_embeds=torch.cat([cur_new_input_embeds,cur_image_features[0:0]],dim=0)
            
            if not chart_token_exist and "chart" in multi_modal_features:
                cur_chart_features = multi_modal_features["chart"][batch_idx]
                cur_new_input_embeds=torch.cat([cur_new_input_embeds,cur_chart_features[0:0]],dim=0)
                
            new_input_embeds.append(cur_new_input_embeds)
            if labels is not None:
                cur_new_labels = torch.cat(cur_new_labels, dim=0)
                new_labels.append(cur_new_labels)
                
            
                
                
        
        """ 
            2.3 batch 내 example 간 modality의 구성이 다른 경우 -> example 간 길이 달라짐
                - 이를 반영 하여 padding 등의 작업 수행
        """
        if any(x.shape != new_input_embeds[0].shape for x in new_input_embeds):
            max_len = max(x.shape[0] for x in new_input_embeds)

            new_input_embeds_align = []
            for cur_new_embed in new_input_embeds:
                cur_new_embed = torch.cat((cur_new_embed, torch.zeros((max_len - cur_new_embed.shape[0], cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device)), dim=0)
                new_input_embeds_align.append(cur_new_embed)
            new_input_embeds = torch.stack(new_input_embeds_align, dim=0)

            if labels is not None:
                new_labels_align = []
                _new_labels = new_labels
                for cur_new_label in new_labels:
                    cur_new_label = torch.cat((cur_new_label, torch.full((max_len - cur_new_label.shape[0],), IGNORE_INDEX, dtype=cur_new_label.dtype, device=cur_new_label.device)), dim=0)
                    new_labels_align.append(cur_new_label)
                new_labels = torch.stack(new_labels_align, dim=0)

            if attention_mask is not None:
                new_attention_mask = []
                for cur_attention_mask, cur_new_labels, cur_new_labels_align in zip(attention_mask, _new_labels, new_labels):
                    new_attn_mask_pad_left = torch.full((cur_new_labels.shape[0] - labels.shape[1],), True, dtype=attention_mask.dtype, device=attention_mask.device)
                    new_attn_mask_pad_right = torch.full((cur_new_labels_align.shape[0] - cur_new_labels.shape[0],), False, dtype=attention_mask.dtype, device=attention_mask.device)
                    cur_new_attention_mask = torch.cat((new_attn_mask_pad_left, cur_attention_mask, new_attn_mask_pad_right), dim=0)
                    new_attention_mask.append(cur_new_attention_mask)
                attention_mask = torch.stack(new_attention_mask, dim=0)
                assert attention_mask.shape == new_labels.shape
        else:
            new_input_embeds = torch.stack(new_input_embeds, dim=0)
            if labels is not None:
                new_labels  = torch.stack(new_labels, dim=0)

            if attention_mask is not None:
                new_attn_mask_pad_left = torch.full((attention_mask.shape[0], new_input_embeds.shape[1] - input_ids.shape[1]), True, dtype=attention_mask.dtype, device=attention_mask.device)
                attention_mask = torch.cat((new_attn_mask_pad_left, attention_mask), dim=1)
                assert attention_mask.shape == new_input_embeds.shape[:2]

        return None, attention_mask, past_key_values, new_input_embeds, new_labels

    def initialize_vision_tokenizer(self, model_args, tokenizer):
        if model_args.mm_use_im_patch_token:
            tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

        if model_args.mm_use_im_start_end:
            print("skip pre implementation of mm_use_im_start_end")
            # num_new_tokens = tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)
            # self.resize_token_embeddings(len(tokenizer))

            # if num_new_tokens > 0:
            #     input_embeddings = self.get_input_embeddings().weight.data
            #     output_embeddings = self.get_output_embeddings().weight.data

            #     input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
            #         dim=0, keepdim=True)
            #     output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
            #         dim=0, keepdim=True)

            #     input_embeddings[-num_new_tokens:] = input_embeddings_avg
            #     output_embeddings[-num_new_tokens:] = output_embeddings_avg

            # if model_args.tune_mm_mlp_adapter:
            #     for p in self.get_input_embeddings().parameters():
            #         p.requires_grad = True
            #     for p in self.get_output_embeddings().parameters():
            #         p.requires_grad = False

            # if model_args.pretrain_mm_mlp_adapter:
            #     mm_projector_weights = torch.load(model_args.pretrain_mm_mlp_adapter, map_location='cpu')
            #     embed_tokens_weight = mm_projector_weights['model.embed_tokens.weight']
            #     assert num_new_tokens == 2
            #     if input_embeddings.shape == embed_tokens_weight.shape:
            #         input_embeddings[-num_new_tokens:] = embed_tokens_weight[-num_new_tokens:]
            #     elif embed_tokens_weight.shape[0] == num_new_tokens:
            #         input_embeddings[-num_new_tokens:] = embed_tokens_weight
            #     else:
            #         raise ValueError(f"Unexpected embed_tokens_weight shape. Pretrained: {embed_tokens_weight.shape}. Current: {input_embeddings.shape}. Numer of new tokens: {num_new_tokens}.")
        elif model_args.mm_use_im_patch_token:
            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = False
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False
