# Adopted from https://github.com/lm-sys/FastChat. Below is the original copyright:
# Adopted from tatsu-lab@stanford_alpaca. Below is the original copyright:
#    Copyright 2023 Rohan Taori, Ishaan Gulrajani, Tianyi Zhang, Yann Dubois, Xuechen Li
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.

import os
import copy
from dataclasses import dataclass, field
import json
import logging
import pathlib
from typing import Dict, Optional, Sequence, List
from datasets import load_dataset
import traceback

import torch

import transformers
from transformers import DonutProcessor,  EvalPrediction,AutoModelForCausalLM

from llava.constants import *
from torch.utils.data import Dataset
from llava.train.llava_trainer import LLaVATrainer

from llava import conversation as conversation_lib
# from llava.model import *
from llava.model import LlavaLlamaForCausalLM
from llava.mm_utils import tokenizer_modal_token, tokenizer_multi_modal_token

import random
from PIL import Image


local_rank = None

os.environ["WANDB__SERVICE_WAIT"] = "300"

def rank0_print(*args):
    if local_rank == 0:
        print(*args)


@dataclass
class ModelArguments:
    # load_wo_init_vision True -> llava 같은 모델 로딩할 때. projection layer 따로 init하지 않음
    # load_wo_init_vision False / pretrain_mm_mlp_adapter -> vicuna 같은 모델과 pretrained 모델 로딩 할때. projection layer init할 때 pretrain_mm_mlp_adapter 사용
    load_wo_init_vision: bool=field(default=False) # projection layer를 따로 init하지 않을 때 사용 (ex) llava를 loading할떄는 vision projection layer를 loading 하므로 init이 필요없음)
    load_wo_init_chart: bool=field(default=False)
    model_name_or_path: Optional[str] = field(default="facebook/opt-125m")
    
    version: Optional[str] = field(default="v0")
    freeze_backbone: bool = field(default=False)
    
    # vision modality
    vision_tower: Optional[str] = field(default=None)
    mm_vision_select_layer: Optional[int] = field(default=-2)   # 몇번째 마지막 layer가 사용되는지
    mm_vision_select_feature: Optional[str] = field(default="patch") # CLIPVisionTower의 parameter
    pretrain_mm_mlp_adapter: Optional[str] = field(default=None) # vision projection layer weight 경로
    mm_projector_type: Optional[str] = field(default='mlp2x_gelu') # projection layer type
    
    # chart modality
    chart_tower: Optional[str] = field(default=None)
    mm_chart_select_layer: Optional[int] = field(default=-2) 
    mm_chart_projector_type: Optional[str] = field(default='mlp2x_gelu')
    pretrain_chart_mm_mlp_adapter: Optional[str] = field(default=None)
    # tune_chart_mm_mlp_adapter: bool = field(default=False)
    
    tune_mm_mlp_adapter: bool = field(default=False)  # 모든 projection layer tuning 여부 / True일 경우 모델은 학습하지 않고 projection layer만 학습!
    
    mm_use_chart_start_end: bool = field(default=False)# <chart_start>, <chart_end> 사용
    use_special_token_task: bool = field(default=False) # special token 사용 task별로 <summarize_chart>, <chart_qa>
    # Disabled
    mm_use_im_start_end: bool = field(default=False) # <im_start>, <im_end> token 사용
    mm_use_im_patch_token: bool = field(default=False) # <im_patch 사용>
    


@dataclass
class DataArguments:
    data_type: str = field(default="image",metadata={"help":'["chart", "image"]'})
    train_data_path: str = field(default=None,metadata={"help": "Path to the training data."})
    
    eval_data_path: str = field(default=None,metadata={"help": "Path to the validataion data."})
    # pred_data_path: str = field(default=None,metadata={"help": "Path to the prediction data."})
    
    lazy_preprocess: bool = False
    is_multimodal: bool = False
    image_folder: Optional[str] = field(default=None)
    chart_folder:Optional[str] = field(default=None)
    image_aspect_ratio: str = 'square'
    image_grid_pinpoints: Optional[str] = field(default=None)
    
    image_processor = None
    chart_processor =None


@dataclass
class TrainingArguments(transformers.TrainingArguments):
    cache_dir: Optional[str] = field(default=None)
    optim: str = field(default="adamw_torch")
    remove_unused_columns: bool = field(default=False)
    freeze_mm_mlp_adapter: bool = field(default=False) # 모든 Modality의 projection layer freezing!!!
    mpt_attn_impl: Optional[str] = field(default="triton")
    model_max_length: int = field(
        default=512,
        metadata={
            "help":
            "Maximum sequence length. Sequences will be right padded (and possibly truncated)."
        },
    )
    double_quant: bool = field(
        default=True,
        metadata={"help": "Compress the quantization statistics through double quantization."}
    )
    quant_type: str = field(
        default="nf4",
        metadata={"help": "Quantization data type to use. Should be one of `fp4` or `nf4`."}
    )
    bits: int = field(
        default=16,
        metadata={"help": "How many bits to use."}
    )
    lora_enable: bool = False
    lora_r: int = 64
    lora_alpha: int = 16
    lora_dropout: float = 0.05
    lora_weight_path: str = ""
    lora_bias: str = "none"
    group_by_modality_length: bool = field(default=False)


def maybe_zero_3(param, ignore_status=False, name=None):
    from deepspeed import zero
    from deepspeed.runtime.zero.partition_parameters import ZeroParamStatus
    if hasattr(param, "ds_id"):
        if param.ds_status == ZeroParamStatus.NOT_AVAILABLE:
            if not ignore_status:
                logging.warning(f"{name}: param.ds_status != ZeroParamStatus.NOT_AVAILABLE: {param.ds_status}")
        with zero.GatheredParameters([param]):
            param = param.data.detach().cpu().clone()
    else:
        param = param.detach().cpu().clone()
    return param


# Borrowed from peft.utils.get_peft_model_state_dict
def get_peft_state_maybe_zero_3(named_params, bias):
    if bias == "none":
        to_return = {k: t for k, t in named_params if "lora_" in k}
    elif bias == "all":
        to_return = {k: t for k, t in named_params if "lora_" in k or "bias" in k}
    elif bias == "lora_only":
        to_return = {}
        maybe_lora_bias = {}
        lora_bias_names = set()
        for k, t in named_params:
            if "lora_" in k:
                to_return[k] = t
                bias_name = k.split("lora_")[0] + "bias"
                lora_bias_names.add(bias_name)
            elif "bias" in k:
                maybe_lora_bias[k] = t
        for k, t in maybe_lora_bias:
            if bias_name in lora_bias_names:
                to_return[bias_name] = t
    else:
        raise NotImplementedError
    to_return = {k: maybe_zero_3(v, ignore_status=True) for k, v in to_return.items()}
    return to_return


def get_peft_state_non_lora_maybe_zero_3(named_params, require_grad_only=True):
    to_return = {k: t for k, t in named_params if "lora_" not in k}
    if require_grad_only:
        to_return = {k: t for k, t in to_return.items() if t.requires_grad}
    to_return = {k: maybe_zero_3(v, ignore_status=True).cpu() for k, v in to_return.items()}
    return to_return


def get_mm_adapter_state_maybe_zero_3(named_params, keys_to_match):
    to_return = {k: t for k, t in named_params if any(key_match in k for key_match in keys_to_match)}
    to_return = {k: maybe_zero_3(v, ignore_status=True).cpu() for k, v in to_return.items()}
    return to_return


def find_all_linear_names(model):
    cls = torch.nn.Linear
    lora_module_names = set()
    multimodal_keywords = ['mm_projector', 'vision_tower', 'vision_resampler']
    for name, module in model.named_modules():
        if any(mm_keyword in name for mm_keyword in multimodal_keywords):
            continue
        if isinstance(module, cls):
            names = name.split('.')
            lora_module_names.add(names[0] if len(names) == 1 else names[-1])

    if 'lm_head' in lora_module_names: # needed for 16-bit
        lora_module_names.remove('lm_head')
    return list(lora_module_names)


def safe_save_model_for_hf_trainer(trainer: transformers.Trainer,
                                   output_dir: str):
    """Collects the state dict and dump to disk."""

    if getattr(trainer.args, "tune_mm_mlp_adapter", False):
        # Only save Adapter
        keys_to_match = ['mm_projector']
        if getattr(trainer.args, "use_im_start_end", False):
            keys_to_match.extend(['embed_tokens', 'embed_in'])

        weight_to_save = get_mm_adapter_state_maybe_zero_3(trainer.model.named_parameters(), keys_to_match)
        trainer.model.config.save_pretrained(output_dir)

        current_folder = output_dir.split('/')[-1]
        parent_folder = os.path.dirname(output_dir)
        if trainer.args.local_rank == 0 or trainer.args.local_rank == -1:
            if current_folder.startswith('checkpoint-'):
                mm_projector_folder = os.path.join(parent_folder, "mm_projector")
                os.makedirs(mm_projector_folder, exist_ok=True)
                torch.save(weight_to_save, os.path.join(mm_projector_folder, f'{current_folder}.bin'))
            else:
                torch.save(weight_to_save, os.path.join(output_dir, f'mm_projector.bin'))
        return

    if trainer.deepspeed:
        torch.cuda.synchronize()
        trainer.save_model(output_dir)
        return

    state_dict = trainer.model.state_dict()
    if trainer.args.should_save:
        cpu_state_dict = {
            key: value.cpu()
            for key, value in state_dict.items()
        }
        del state_dict
        trainer._save(output_dir, state_dict=cpu_state_dict)  # noqa


def smart_tokenizer_and_embedding_resize(
    special_tokens_dict: Dict,
    tokenizer: transformers.PreTrainedTokenizer,
    model: transformers.PreTrainedModel,
):
    """Resize tokenizer and embedding.

    Note: This is the unoptimized version that may make your embedding size not be divisible by 64.
    """
    num_new_tokens = tokenizer.add_special_tokens(special_tokens_dict)
    model.resize_token_embeddings(len(tokenizer))

    if num_new_tokens > 0:
        input_embeddings = model.get_input_embeddings().weight.data
        output_embeddings = model.get_output_embeddings().weight.data

        input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
            dim=0, keepdim=True)
        output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
            dim=0, keepdim=True)

        input_embeddings[-num_new_tokens:] = input_embeddings_avg
        output_embeddings[-num_new_tokens:] = output_embeddings_avg


def _tokenize_fn(strings: Sequence[str],
                 tokenizer: transformers.PreTrainedTokenizer) -> Dict:
    """Tokenize a list of strings."""
    tokenized_list = [
        tokenizer(
            text,
            return_tensors="pt",
            padding="longest",
            max_length=tokenizer.model_max_length,
            truncation=True,
        ) for text in strings
    ]
    input_ids = labels = [
        tokenized.input_ids[0] for tokenized in tokenized_list
    ]
    input_ids_lens = labels_lens = [
        tokenized.input_ids.ne(tokenizer.pad_token_id).sum().item()
        for tokenized in tokenized_list
    ]
    return dict(
        input_ids=input_ids,
        labels=labels,
        input_ids_lens=input_ids_lens,
        labels_lens=labels_lens,
    )


def _mask_targets(target, tokenized_lens, speakers):
    # cur_idx = 0
    cur_idx = tokenized_lens[0]
    tokenized_lens = tokenized_lens[1:]
    target[:cur_idx] = IGNORE_INDEX
    for tokenized_len, speaker in zip(tokenized_lens, speakers):
        if speaker == "human":
            target[cur_idx+2:cur_idx + tokenized_len] = IGNORE_INDEX
        cur_idx += tokenized_len


def _add_speaker_and_signal(header, source, get_conversation=True):
    """Add speaker and start/end signal on each round."""
    BEGIN_SIGNAL = "### "
    END_SIGNAL = "\n"
    conversation = header
    for sentence in source:
        from_str = sentence["from"]
        if from_str.lower() == "human":
            from_str = conversation_lib.default_conversation.roles[0]
        elif from_str.lower() == "gpt":
            from_str = conversation_lib.default_conversation.roles[1]
        else:
            from_str = 'unknown'
        sentence["value"] = (BEGIN_SIGNAL + from_str + ": " +
                             sentence["value"] + END_SIGNAL)
        if get_conversation:
            conversation += sentence["value"]
    conversation += BEGIN_SIGNAL
    return conversation


def preprocess_multimodal(
    sources: Sequence[str],
    data_args: DataArguments
) -> Dict:
    # <image>, <chart> Tag를 맨 앞으로 옮기고 \n 붙임
    # ex) 'Share a concise interpretation of the image provided.<Image>' -> <image>\nShare a concise interpretation of the image provided.
    is_multimodal = data_args.is_multimodal
    if not is_multimodal:
        return sources
    
    def _modal_token_to_front(modal_token, sentence):
        if modal_token in sentence:
            sentence = sentence.replace(modal_token+"\n", '').strip()
            sentence = sentence.replace(modal_token, '').strip()
            sentence = modal_token + '\n' + sentence
            sentence = sentence.strip()
        return sentence
        

    for source in sources:
        for sentence in source:
            sentence['value']=_modal_token_to_front(DEFAULT_IMAGE_TOKEN,sentence['value'])
            sentence['value']=_modal_token_to_front(DEFAULT_CHART_TOKEN,sentence['value'])
            
            # if DEFAULT_CHART_TOKEN in sentence['value']:
            #     sentence['value'] = sentence['value'].replace(DEFAULT_IMAGE_TOKEN, '').strip()
            # if DEFAULT_IMAGE_TOKEN in sentence['value']:
            #     sentence['value'] = sentence['value'].replace(DEFAULT_IMAGE_TOKEN, '').strip()
            #     sentence['value'] = DEFAULT_IMAGE_TOKEN + '\n' + sentence['value']
            #     sentence['value'] = sentence['value'].strip()
            #     if "mmtag" in conversation_lib.default_conversation.version:
            #         sentence['value'] = sentence['value'].replace(DEFAULT_IMAGE_TOKEN, '<Image>' + DEFAULT_IMAGE_TOKEN + '</Image>')
            # replace_token = DEFAULT_IMAGE_TOKEN
            # if data_args.mm_use_im_start_end:
            #     replace_token = DEFAULT_IM_START_TOKEN + replace_token + DEFAULT_IM_END_TOKEN
            # sentence["value"] = sentence["value"].replace(DEFAULT_IMAGE_TOKEN, replace_token)

    return sources


def preprocess_llama_2(
    sources,
    tokenizer: transformers.PreTrainedTokenizer,
    has_image: bool = False
) -> Dict:
    conv = conversation_lib.default_conversation.copy()
    roles = {"human": conv.roles[0], "gpt": conv.roles[1]}

    # Apply prompt templates
    conversations = []
    for i, source in enumerate(sources):
        if roles[source[0]["from"]] != conv.roles[0]:
            # Skip the first one if it is not from human
            source = source[1:]

        conv.messages = []
        for j, sentence in enumerate(source):
            role = roles[sentence["from"]]
            assert role == conv.roles[j % 2], f"{i}"
            conv.append_message(role, sentence["value"])
        conversations.append(conv.get_prompt())

    # Tokenize conversations

    if has_image:
        input_ids = torch.stack([tokenizer_modal_token(prompt, tokenizer, return_tensors='pt') for prompt in conversations], dim=0)
    else:
        input_ids = tokenizer(
            conversations,
            return_tensors="pt",
            padding="longest",
            max_length=tokenizer.model_max_length,
            truncation=True,
        ).input_ids

    targets = input_ids.clone()

    assert conv.sep_style == conversation_lib.SeparatorStyle.LLAMA_2

    # Mask targets
    sep = "[/INST] "
    for conversation, target in zip(conversations, targets):
        total_len = int(target.ne(tokenizer.pad_token_id).sum())

        rounds = conversation.split(conv.sep2)
        cur_len = 1
        target[:cur_len] = IGNORE_INDEX
        for i, rou in enumerate(rounds):
            if rou == "":
                break

            parts = rou.split(sep)
            if len(parts) != 2:
                break
            parts[0] += sep

            if has_image:
                round_len = len(tokenizer_modal_token(rou, tokenizer))
                instruction_len = len(tokenizer_modal_token(parts[0], tokenizer)) - 2
            else:
                round_len = len(tokenizer(rou).input_ids)
                instruction_len = len(tokenizer(parts[0]).input_ids) - 2

            target[cur_len : cur_len + instruction_len] = IGNORE_INDEX

            cur_len += round_len
        target[cur_len:] = IGNORE_INDEX

        if cur_len < tokenizer.model_max_length:
            if cur_len != total_len:
                target[:] = IGNORE_INDEX
                print(
                    f"WARNING: tokenization mismatch: {cur_len} vs. {total_len}."
                    f" (ignored)"
                )

    return dict(
        input_ids=input_ids,
        labels=targets,
    )

def preprocess_llama_3(
    sources,
    tokenizer: transformers.PreTrainedTokenizer
) -> Dict:
    conv = conversation_lib.conv_llama_3.copy()
    roles = {"human": conv.roles[0], "gpt": conv.roles[1]}

    has_image=False
    has_chart=False
    
    for source in sources:
        for sentence in source:
            if "<image>" in sentence["value"]:
                has_image=True
                modality_token="<image>"
            if "<chart>" in sentence["value"]:
                has_chart=True
                modality_token="<chart>"
                
    # Apply prompt templates
    conversations = []
    for i, source in enumerate(sources):
        if roles[source[0]["from"]] != conv.roles[0]:
            # Skip the first one if it is not from human
            source = source[1:]

        conv.messages = []
        for j, sentence in enumerate(source):
            role = roles[sentence["from"]]
            assert role == conv.roles[j % 2], f"{i}"
            conv.append_message(role, sentence["value"])
        conversations.append(conv.get_prompt())
        
    # Tokenize conversations
    if has_image or has_chart:
        # input_ids = torch.stack([tokenizer_modal_token(prompt, tokenizer,modality_token =modality_token,return_tensors='pt') for prompt in conversations], dim=0)
        input_ids = torch.stack([tokenizer_multi_modal_token(prompt, tokenizer) for prompt in conversations], dim=0)
    # if has_image:
    #     input_ids = torch.stack([tokenizer_image_token(prompt, tokenizer, return_tensors='pt') for prompt in conversations], dim=0)
    else:
        input_ids = tokenizer(
            conversations,
            return_tensors="pt",
            padding="longest",
            max_length=tokenizer.model_max_length,
            truncation=True,
        ).input_ids

    targets = input_ids.clone()
    
    offset = 0 if input_ids[0][0] != tokenizer.bos_token_id else 1
    
    assert conv.sep_style == conversation_lib.SeparatorStyle.TWO
    # Mask targets
    sep = conv.sep + conv.roles[1] + ":"
    # Llama3 tokenizer has the token for whitespace
    # Typically, the token after whitespace will be naturally encoded as one token with whitespace
    # some special cases like ": 3" will be encoded as :, whitespace, 3; 3 tokens. Only in this case, the loss on whitespace will be calculated
    for conversation, target in zip(conversations, targets):
        total_len = int(target.ne(tokenizer.pad_token_id).sum())

        rounds = conversation.split(conv.sep2)
        cur_len = 1
        target[:cur_len] = IGNORE_INDEX
        
        for i, rou in enumerate(rounds):
            if rou == "":
                break

            parts = rou.split(sep)
            if len(parts) != 2:
                break
            parts[0] += sep

            if has_image:
                round_len = len(tokenizer_multi_modal_token(rou, tokenizer))
                instruction_len = len(tokenizer_multi_modal_token(parts[0], tokenizer)) - offset
            else:
                round_len = len(tokenizer(rou).input_ids)
                instruction_len = len(tokenizer(parts[0]).input_ids) - offset

            target[cur_len : cur_len + instruction_len] = IGNORE_INDEX

            cur_len += round_len + (1 - offset) #starting from index 0, then cur_len will not cover eos token

        if cur_len < tokenizer.model_max_length:
            if cur_len != total_len:
                target[:] = IGNORE_INDEX
                print(
                    f"WARNING: tokenization mismatch: {cur_len} vs. {total_len}."
                    f" (ignored)"
                )

    if input_ids[0][0] != tokenizer.bos_token_id:
        input_ids = [torch.cat([torch.LongTensor([tokenizer.bos_token_id]), i]) for i in input_ids]
        targets = [torch.cat([torch.LongTensor([IGNORE_INDEX]), i]) for i in targets]

    return dict(
        input_ids=input_ids,
        labels=targets,
    )

def preprocess_v1(
    sources,
    tokenizer: transformers.PreTrainedTokenizer,
    system_prompt=None
) -> Dict:
    conv = conversation_lib.default_conversation.copy()
    if system_prompt!=None:
        conv.system=system_prompt
        
    roles = {"human": conv.roles[0], "gpt": conv.roles[1]}
    """
        - System prompt 추가하여 prompt 생성 후 Tokenization
            - modality token(<chart>,<image>)을 modality special token으로 대체
            - input_ids, targets(-100으로 instruction 부분 masking) 생성

    """
    
    has_image=False
    has_chart=False
    
    
    for source in sources:
        for sentence in source:
            if "<image>" in sentence["value"]:
                has_image=True
                modality_token="<image>"
            if "<chart>" in sentence["value"]:
                has_chart=True
                modality_token="<chart>"
                
    """ 
        1. Apply prompt templates # {SYSTEM} USER: ~~~ ASSISTANT: ~~~ </s>
    """
    conversations = []
    for i, source in enumerate(sources):
        if roles[source[0]["from"]] != conv.roles[0]:
            # Skip the first one if it is not from human
            source = source[1:]

        conv.messages = []
        for j, sentence in enumerate(source):
            role = roles[sentence["from"]]
            assert role == conv.roles[j % 2], f"{i}"
            conv.append_message(role, sentence["value"])
        conversations.append(conv.get_prompt())

    """
        2. Tokenize conversations
            - <chart>, <image>를 각각 CHART_TOKEN_INDEX, IMAGE_TOKEN_INDEX로 replace
    """
    if has_image or has_chart:
        # input_ids = torch.stack([tokenizer_modal_token(prompt, tokenizer,modality_token =modality_token,return_tensors='pt') for prompt in conversations], dim=0)
        input_ids = torch.stack([tokenizer_multi_modal_token(prompt, tokenizer) for prompt in conversations], dim=0)

    else:
        input_ids = tokenizer(
            conversations,
            return_tensors="pt",
            padding="longest",
            max_length=tokenizer.model_max_length,
            truncation=True,
        ).input_ids

    targets = input_ids.clone()

    assert conv.sep_style == conversation_lib.SeparatorStyle.TWO

    """
        3. Mask targets( Instruction 부분 -100으로 Masking) / Multi-turn 고려
    """

    sep = conv.sep + conv.roles[1] + ": "
    for conversation, target in zip(conversations, targets):
        total_len = int(target.ne(tokenizer.pad_token_id).sum())

        rounds = conversation.split(conv.sep2)
        cur_len = 1
        target[:cur_len] = IGNORE_INDEX
        for i, rou in enumerate(rounds):
            if rou == "":
                break

            parts = rou.split(sep)
            if len(parts) != 2:
                break
            parts[0] += sep

            if has_image or has_chart:
                round_len = len(tokenizer_multi_modal_token(rou,tokenizer))
                instruction_len = len(tokenizer_multi_modal_token(parts[0],tokenizer))-2
                # round_len = len(tokenizer_modal_token(rou, tokenizer,modality_token =modality_token))
                # instruction_len = len(tokenizer_modal_token(parts[0], tokenizer,modality_token =modality_token)) - 2
            else:
                round_len = len(tokenizer(rou).input_ids)
                instruction_len = len(tokenizer(parts[0]).input_ids) - 2

            target[cur_len : cur_len + instruction_len] = IGNORE_INDEX

            cur_len += round_len
        target[cur_len:] = IGNORE_INDEX

        if cur_len < tokenizer.model_max_length:
            if cur_len != total_len:
                target[:] = IGNORE_INDEX
                print(
                    f"WARNING: tokenization mismatch: {cur_len} vs. {total_len}."
                    f" (ignored)"
                )
    # temp=[x for x in targets.squeeze().tolist() if x!=-100]
    # print(tokenizer.convert_ids_to_tokens(temp))
    return dict(
        input_ids=input_ids,
        labels=targets,
    )


def preprocess_mpt(
    sources,
    tokenizer: transformers.PreTrainedTokenizer,
) -> Dict:
    conv = conversation_lib.default_conversation.copy()
    roles = {"human": conv.roles[0], "gpt": conv.roles[1]}

    # Apply prompt templates
    conversations = []
    for i, source in enumerate(sources):
        if roles[source[0]["from"]] != conv.roles[0]:
            # Skip the first one if it is not from human
            source = source[1:]

        conv.messages = []
        for j, sentence in enumerate(source):
            role = roles[sentence["from"]]
            assert role == conv.roles[j % 2], f"{i}"
            conv.append_message(role, sentence["value"])
        conversations.append(conv.get_prompt())

    # Tokenize conversations
    input_ids = torch.stack([tokenizer_modal_token(prompt, tokenizer, return_tensors='pt') for prompt in conversations], dim=0)
    targets = input_ids.clone()
    assert conv.sep_style == conversation_lib.SeparatorStyle.MPT

    # Mask targets
    sep = conv.sep + conv.roles[1]
    for conversation, target in zip(conversations, targets):
        total_len = int(target.ne(tokenizer.pad_token_id).sum())

        rounds = conversation.split(conv.sep)
        re_rounds = [conv.sep.join(rounds[:3])] # system + user + gpt
        for conv_idx in range(3, len(rounds), 2):
            re_rounds.append(conv.sep.join(rounds[conv_idx:conv_idx+2]))    # user + gpt
        cur_len = 0
        target[:cur_len] = IGNORE_INDEX
        for i, rou in enumerate(re_rounds):
            if rou == "":
                break

            parts = rou.split(sep)
            if len(parts) != 2:
                break
            parts[0] += sep
            round_len = len(tokenizer_modal_token(rou, tokenizer)) + len(tokenizer_modal_token(conv.sep, tokenizer))
            instruction_len = len(tokenizer_modal_token(parts[0], tokenizer))
            target[cur_len : cur_len + instruction_len] = IGNORE_INDEX

            cur_len += round_len
        target[cur_len:] = IGNORE_INDEX

        if cur_len < tokenizer.model_max_length:
            if cur_len != total_len:
                target[:] = IGNORE_INDEX
                print(
                    f"WARNING: tokenization mismatch: {cur_len} vs. {total_len}."
                    f" (ignored)"
                )

    return dict(
        input_ids=input_ids,
        labels=targets,
    )



def preprocess_plain_v2(
    sources: Sequence[str],
    tokenizer: transformers.PreTrainedTokenizer,
) -> Dict:
    # LLM Input = <Image> caption, Label = Caption으로 만듬 (x)
    # LLM Input = 원본 그대로 caption Label = Caption으로 만듬
    # <Image>\nprompt\nlabel</s>
    # add end signal and concatenate together
    modality_token=""
    conversations = []
    for source in sources:
        assert len(source) == 2
        assert DEFAULT_IMAGE_TOKEN in source[0]['value'] or DEFAULT_CHART_TOKEN in source[0]['value']
        if DEFAULT_IMAGE_TOKEN in source[0]['value']:
            # source[0]['value'] = DEFAULT_IMAGE_TOKEN
            modality_token="<image>"
        elif DEFAULT_CHART_TOKEN in source[0]['value']:
            # source[0]['value'] = DEFAULT_CHART_TOKEN
            modality_token="<chart>"
        conversation = source[0]['value'] + " " +source[1]['value'] + conversation_lib.default_conversation.sep
        conversations.append(conversation)
    # tokenize conversations
    # input_ids = [tokenizer_modal_token(prompt, tokenizer, modality_token=modality_token, return_tensors='pt') for prompt in conversations]
    input_ids = [tokenizer_multi_modal_token(prompt, tokenizer) for prompt in conversations]
    # input_ids = [tokenizer_modal_token(prompt, tokenizer, modality_token=modality_token, return_tensors='pt') for prompt in conversations]

    targets = copy.deepcopy(input_ids)
    for target, source in zip(targets, sources):
        # tokenized_len = len(tokenizer_modal_token(source[0]['value'], tokenizer,modality_token=modality_token))
        tokenized_len = len(tokenizer_multi_modal_token(source[0]['value'], tokenizer))
        target[:tokenized_len] = IGNORE_INDEX

    return dict(input_ids=input_ids, labels=targets)


def preprocess_plain(
    sources: Sequence[str],
    tokenizer: transformers.PreTrainedTokenizer,
) -> Dict:
    # LLM Input = <Image> caption, Label = Caption으로 만듬 
    # add end signal and concatenate together
    modality_token=""
    conversations = []
    for source in sources:
        assert len(source) == 2
        assert DEFAULT_IMAGE_TOKEN in source[0]['value'] or DEFAULT_CHART_TOKEN in source[0]['value']
        if DEFAULT_IMAGE_TOKEN in source[0]['value']:
            source[0]['value'] = DEFAULT_IMAGE_TOKEN
            modality_token="<image>"
        elif DEFAULT_CHART_TOKEN in source[0]['value']:
            source[0]['value'] = DEFAULT_CHART_TOKEN
            modality_token="<chart>"
        conversation = source[0]['value'] + source[1]['value'] + conversation_lib.default_conversation.sep
        conversations.append(conversation)
    # tokenize conversations
    
    input_ids = [tokenizer_modal_token(prompt, tokenizer, modality_token=modality_token, return_tensors='pt') for prompt in conversations]
    targets = copy.deepcopy(input_ids)
    for target, source in zip(targets, sources):
        tokenized_len = len(tokenizer_modal_token(source[0]['value'], tokenizer,modality_token=modality_token))
        target[:tokenized_len] = IGNORE_INDEX

    return dict(input_ids=input_ids, labels=targets)


def preprocess(
    sources: Sequence[str],
    tokenizer: transformers.PreTrainedTokenizer,
    system_prompt=None
) -> Dict:
    """
    Given a list of sources, each is a conversation list. This transform:
    1. Add signal '### ' at the beginning each sentence, with end signal '\n';
    2. Concatenate conversations together;
    3. Tokenize the concatenated conversation;
    4. Make a deepcopy as the target. Mask human words with IGNORE_INDEX.
    """
    
    # version 등에 따라 다르게 동작하도록 구현한듯?
    if conversation_lib.default_conversation.sep_style == conversation_lib.SeparatorStyle.PLAIN:
        return preprocess_plain(sources, tokenizer)
    elif conversation_lib.default_conversation.version.startswith("v3"): # for llama 3 tokenizer
        return preprocess_llama_3(sources, tokenizer)
    elif conversation_lib.default_conversation.sep_style == conversation_lib.SeparatorStyle.PLAIN_V2:
        return preprocess_plain_v2(sources, tokenizer)
    elif conversation_lib.default_conversation.version.startswith("v1"):
        return preprocess_v1(sources, tokenizer,system_prompt)
    else:
        print("Other version not implemented yet")
        raise NotImplementedError



class LazySupervisedDataset(Dataset):
    """Dataset for supervised fine-tuning."""

    def __init__(self, data_path: str,
                 tokenizer: transformers.PreTrainedTokenizer,
                 data_args: DataArguments):
        super(LazySupervisedDataset, self).__init__()
        list_data_dict = json.load(open(data_path, "r"))
        
        print("Shuffling data")
        random.seed(1000)
        random.shuffle(list_data_dict)

        if  "system_prompt" in list_data_dict[0]:
            rank0_print(f"System prompt int data will be used\n For example, \"{list_data_dict[0]['system_prompt']}\" ")


        rank0_print("Formatting inputs...Skip in lazy mode")
        self.tokenizer = tokenizer
        self.list_data_dict = list_data_dict
        self.data_args = data_args

    def __len__(self):
        return len(self.list_data_dict)

    @property
    def lengths(self):
        length_list = []
        for sample in self.list_data_dict:
            img_tokens = 128 if 'image' in sample else 0
            length_list.append(sum(len(conv['value'].split()) for conv in sample['conversations']) + img_tokens)
        return length_list

    @property
    def modality_lengths(self):
        length_list = []
        for sample in self.list_data_dict:
            cur_len = sum(len(conv['value'].split()) for conv in sample['conversations'])
            cur_len = cur_len if 'image' in sample else -cur_len
            length_list.append(cur_len)
        return length_list

    def __getitem__(self, i) -> Dict[str, torch.Tensor]:
        sources = self.list_data_dict[i]
        if isinstance(i, int):
            sources = [sources]
        assert len(sources) == 1, "Don't know why it is wrapped to a list"  # FIXME
        
        # print(f'getitem {i}, / {sources[0]["conversations"][0]["value"]}')
        """
            !! Modality 추가시 구현 필요 부분 !!
        """
        
        try:
            """
            1. Multimodal 데이터 기본 전처리
            """
            if 'image' in sources[0]:
                image_file = self.list_data_dict[i]['image']
                image_folder = self.data_args.image_folder
                processor = self.data_args.image_processor
                image = Image.open(os.path.join(image_folder, image_file)).convert('RGB')
                if self.data_args.image_aspect_ratio == 'pad':
                    def expand2square(pil_img, background_color):
                        width, height = pil_img.size
                        if width == height:
                            return pil_img
                        elif width > height:
                            result = Image.new(pil_img.mode, (width, width), background_color)
                            result.paste(pil_img, (0, (width - height) // 2))
                            return result
                        else:
                            result = Image.new(pil_img.mode, (height, height), background_color)
                            result.paste(pil_img, ((height - width) // 2, 0))
                            return result
                    image = expand2square(image, tuple(int(x*255) for x in processor.image_mean))
                    image = processor.preprocess(image, return_tensors='pt')['pixel_values'][0] 
                else:
                    # ClipImageProcessor라면 - https://huggingface.co/docs/transformers/v4.35.0/en/model_doc/clip#transformers.CLIPImageProcessor.preprocess
                    image = processor.preprocess(image, return_tensors='pt')['pixel_values'][0]
                    
                # # <image> Tag를 맨 앞으로 옮기고 \n 붙임
                # sources = preprocess_multimodal(
                #     copy.deepcopy([e["conversations"] for e in sources]),
                #     self.data_args)
                
                
            if "chart" in sources[0]:
                chart_file = self.list_data_dict[i]['chart']
                chart_folder = self.data_args.chart_folder
                processor = self.data_args.chart_processor
            
                
                chart_path = os.path.join(chart_folder, chart_file)
                chart = Image.open(chart_path).convert("RGB")
                chart = processor(chart, return_tensors="pt").pixel_values[0]
                # sources = copy.deepcopy([e["conversations"] for e in sources])
            # else:
            #     sources = copy.deepcopy([e["conversations"] for e in sources])
            # <image>, <chart> Tag를 맨 앞으로 옮기고 \n 붙임
            # sources = preprocess_multimodal(
            #     copy.deepcopy([e["conversations"] for e in sources]),
            #     self.data_args)
            system_prompt = sources[0]["system_prompt"] if "system_prompt" in sources[0] else None
            sources = copy.deepcopy([e["conversations"] for e in sources])
                
            """
                2. Prompt 생성 및 Tokenization
            """
            data_dict = preprocess(
                sources,
                self.tokenizer,
                system_prompt)
            
            if isinstance(i, int):
                data_dict = dict(input_ids=data_dict["input_ids"][0],
                                labels=data_dict["labels"][0])


            """ image, chart 데이터를 다루는 모델이지만 image 혹은 chart 데이터 없는 경우에는 zero vector 추가"""
            # image exist in the data
            if 'image' in self.list_data_dict[i]:
                data_dict["image"] = image
            elif self.data_args.image_processor!=None:
                crop_size = self.data_args.image_processor.crop_size
                data_dict['image'] = torch.zeros(3, crop_size['height'], crop_size['width'])
                

            if "chart" in self.list_data_dict[i]:
                data_dict["chart"] = chart
            elif self.data_args.chart_processor!=None:
                crop_size = self.data_args.chart_processor.image_processor.size
                data_dict['chart'] = torch.zeros(3, crop_size['height'], crop_size['width'])
                
            # elif self.data_args.is_multimodal:
            #     print("the case chart not exist NotImplemented")
            #     raise NotImplementedError
            #     # image does not exist in the data, but the model is multimodal
            #     crop_size = self.data_args.image_processor.crop_size
            #     data_dict['image'] = torch.zeros(3, crop_size['height'], crop_size['width'])
        except Exception:
            print("[ERROR] processing data in getitem")
            print(sources)
            print(traceback.format_exc())
            
        return data_dict


@dataclass
class DataCollatorForSupervisedDataset(object):
    """Collate examples for supervised fine-tuning."""

    tokenizer: transformers.PreTrainedTokenizer

    def __call__(self, instances: Sequence[Dict]) -> Dict[str, torch.Tensor]:
        input_ids, labels = tuple([instance[key] for instance in instances]
                                  for key in ("input_ids", "labels"))
        input_ids = torch.nn.utils.rnn.pad_sequence(
            input_ids,
            batch_first=True,
            padding_value=self.tokenizer.pad_token_id)
        labels = torch.nn.utils.rnn.pad_sequence(labels,
                                                 batch_first=True,
                                                 padding_value=IGNORE_INDEX)
        if len(input_ids)>self.tokenizer.model_max_length:
            print(f"batch len = {len(input_ids.shape[1])} this will be truncated")
        input_ids = input_ids[:, :self.tokenizer.model_max_length]
        labels = labels[:, :self.tokenizer.model_max_length]
        batch = dict(
            input_ids=input_ids,
            labels=labels,
            attention_mask=input_ids.ne(self.tokenizer.pad_token_id)
        )


        # if 'image' in instances[0]:
        #     images = [instance['image'] for instance in instances]
        #     if all(x is not None and x.shape == images[0].shape for x in images):
        #         batch['images'] = torch.stack(images)
        #     else:
        #         batch['images'] = images

        batch_images=[]
        batch_charts=[]
        
        # no_image=True
        # img_shape=None
        # no_chart=True
        # chart_shape=None
        
        # for instance in instances:
        #     if "image" in instance:
        #         img_shape=instance["image"].shape
        #         no_image=False
        #     if "chart" in instance:
        #         no_chart=False
        #         chart_shape=instance["chart"].shape
                
        for instance in instances:
            if "image" in instance:
                batch_images.append(instance["image"])
            # elif not no_image:
            #     batch_images.append(torch.full(img_shape,IGNORE_INDEX,dtype=torch.int))
            # else:
            #     batch_images.append(torch.full(img_shape,0))
            if "chart" in instance:
                batch_charts.append(instance["chart"])
            # elif not no_chart:
            #     batch_charts.append(torch.full(chart_shape,IGNORE_INDEX,dtype=torch.int))
        
        if len(batch_images)!=0:
            batch["images"]=torch.stack(batch_images)
        if len(batch_charts)!=0:
            batch["charts"]=torch.stack(batch_charts)
    


        return batch


def make_supervised_data_module(tokenizer: transformers.PreTrainedTokenizer,
                                data_args) -> Dict:
    """Make dataset and collator for supervised fine-tuning."""
    train_dataset=None
    eval_dataset=None
    
    result={}

    if data_args.train_data_path!=None:
        train_dataset = LazySupervisedDataset(tokenizer=tokenizer,
                                    data_path=data_args.train_data_path,
                                    data_args=data_args)
        result["train_dataset"]=train_dataset
        
    
    if data_args.eval_data_path!=None:
        eval_dataset = LazySupervisedDataset(tokenizer=tokenizer,
                                    data_path=data_args.eval_data_path,
                                    data_args=data_args)
        result["eval_dataset"]=eval_dataset
        

    
    
    data_collator = DataCollatorForSupervisedDataset(tokenizer=tokenizer)
    result["data_collator"]=data_collator
    return result


def train():
    global local_rank

    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments))
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    local_rank = training_args.local_rank
    # compute_dtype = (torch.float16 if training_args.fp16 else (torch.bfloat16 if training_args.bf16 else torch.float32))


    # 코드 고치면서 아직 적용하지 않은 parameter Disable!
    if model_args.mm_use_im_patch_token==True:
        raise NotImplementedError
    
    if (model_args.load_wo_init_chart==True and model_args.pretrain_chart_mm_mlp_adapter!=None) or (model_args.load_wo_init_vision==True and model_args.pretrain_mm_mlp_adapter!=None):
        rank0_print("load_wo_init must be FALSE!!! when pretrain_chart_mm_mlp_adapter or pretrain_mm_mlp_adapter NOT NONE")
        raise NotImplementedError


    if model_args.load_wo_init_vision or model_args.load_wo_init_chart :
        
        model = LlavaLlamaForCausalLM.from_pretrained(model_args.model_name_or_path)
        if model_args.load_wo_init_vision:
            print("Not initializing vision projection module!!!")
            vision_tower= model.get_modal_tower(modal="image")
            print("vision tower delay loading!!!")
            vision_tower.load_model()
            vision_tower.to(device="cuda", dtype=torch.float16)
            if vision_tower!=None : data_args.image_processor = vision_tower.image_processor 
        if model_args.load_wo_init_chart:
            print("Not initializing chart projection module!!!")
            chart_tower = model.get_modal_tower(modal="chart")
            if chart_tower!=None: data_args.chart_processor = chart_tower.chart_processor

    elif model_args.vision_tower is not None or model_args.chart_tower is not None:

            model = LlavaLlamaForCausalLM.from_pretrained(
                model_args.model_name_or_path,
                cache_dir=training_args.cache_dir,
                # **bnb_model_from_pretrained_args
            )
    else:
        model = transformers.LlamaForCausalLM.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            # **bnb_model_from_pretrained_args
        )
    model.config.use_cache = False
    data_args.model=model

    
    if model_args.freeze_backbone:
        model.model.requires_grad_(False)


    if training_args.gradient_checkpointing:
        if hasattr(model, "enable_input_require_grads"):
            model.enable_input_require_grads()
        else:
            def make_inputs_require_grad(module, input, output):
                output.requires_grad_(True)
            model.get_input_embeddings().register_forward_hook(make_inputs_require_grad)

    if training_args.lora_enable:
        from peft import LoraConfig, get_peft_model
        lora_config = LoraConfig(
            r=training_args.lora_r,
            lora_alpha=training_args.lora_alpha,
            target_modules=find_all_linear_names(model),
            lora_dropout=training_args.lora_dropout,
            bias=training_args.lora_bias,
            task_type="CAUSAL_LM",
        )
        if training_args.bits == 16:
            if training_args.bf16:
                model.to(torch.bfloat16)
            if training_args.fp16:
                model.to(torch.float16)
        rank0_print("Adding LoRA adapters...")
        model = get_peft_model(model, lora_config)

    if 'mpt' in model_args.model_name_or_path:
        tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right"
        )
    else:
        tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right",
            # use_fast=False,
        )
        
    """ add new special tokens"""


    add_new_tokens=[]
    if model_args.mm_use_chart_start_end==True:
        rank0_print("USE MODAL START END <chart_start>,<chart_end>)")
        add_new_tokens.extend([DEFAULT_CHART_START_TOKEN, DEFAULT_CHART_END_TOKEN])
        
    if model_args.mm_use_im_start_end==True:
        rank0_print("USE MODAL START END <im_start>,<im_end>)")
        add_new_tokens.extend([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN])
    if model_args.use_special_token_task==True:
        rank0_print("USE speical token task ( <summarize_chart>, <chart_qa>)")
        add_new_tokens.extend([SUMMARIZE_CHART_TOKEN, CHART_QA_TOKEN])
    if len(add_new_tokens):
        num_new_tokens = tokenizer.add_tokens(add_new_tokens, special_tokens=True)
        model.resize_token_embeddings(len(tokenizer))
        
        if num_new_tokens > 0:
            input_embeddings = model.get_input_embeddings().weight.data
            output_embeddings = model.get_output_embeddings().weight.data

            input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
                dim=0, keepdim=True)
            output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
                dim=0, keepdim=True)

            input_embeddings[-num_new_tokens:] = input_embeddings_avg
            output_embeddings[-num_new_tokens:] = output_embeddings_avg


        
    if model_args.version == "v0":
        if tokenizer.pad_token is None:
            smart_tokenizer_and_embedding_resize(
                special_tokens_dict=dict(pad_token="[PAD]"),
                tokenizer=tokenizer,
                model=model,
            )
    elif model_args.version == "v0.5":
        tokenizer.pad_token = tokenizer.unk_token
    elif model_args.version == "v3": # llama3
        print("Model version = 3 / special token 0 will be used for pad token / llama_3 conversations")
        tokenizer.pad_token = "<|reserved_special_token_0|>" # only for llama3
        conversation_lib.default_conversation = conversation_lib.conv_templates["llama_3"]
    else:
        if "llama-3" in model_args.model_name_or_path.lower():
             print("llama-3 special token 0 will be used for pad token")
             tokenizer.pad_token = "<|reserved_special_token_0|>"
        else:
            tokenizer.pad_token = tokenizer.unk_token
            
        if model_args.version in conversation_lib.conv_templates:
            conversation_lib.default_conversation = conversation_lib.conv_templates[model_args.version]
        else:
            conversation_lib.default_conversation = conversation_lib.conv_templates["vicuna_v1"]

        
    # <chart>, <image> special token 추가 (Only for tokenization)
    tokenizer.add_tokens(DEFAULT_CHART_TOKEN,special_tokens=True)
    tokenizer.add_tokens(DEFAULT_IMAGE_TOKEN,special_tokens=True)
    
    """ 
        !! Modality 추가시 구현 필요 부분 !!
        
        - Modality별 Encoder, Projection Layer, processor 초기화 
        - Projection layer 학습 여부 설정
    """
    if (model_args.load_wo_init_vision==False and model_args.vision_tower is not None) or (model_args.load_wo_init_chart==False and model_args.chart_tower is not None):
        
        def _init_projection_layer_train(modal_projector,tune_modal_mm_mlp_adapter,training_args):
            
            # if training_args.bits in [4, 8]:
            #     modal_projector.to(dtype=compute_dtype, device=training_args.device)
            
            if tune_modal_mm_mlp_adapter:
                model.requires_grad_(False)
                for p in modal_projector.parameters():
                    p.requires_grad = True  

            if training_args.freeze_mm_mlp_adapter:
                for p in modal_projector.parameters():
                    p.requires_grad = False
                    
        model.config.freeze_mm_mlp_adapter = training_args.freeze_mm_mlp_adapter    
        data_args.is_multimodal = True
        model.config.tune_mm_mlp_adapter = training_args.tune_mm_mlp_adapter = model_args.tune_mm_mlp_adapter

        if model_args.load_wo_init_vision==False and model_args.vision_tower is not None :
            rank0_print(f"INIT Vision VisionTower = {model_args.vision_tower}")
            model.get_model().initialize_modal_modules(
                model_args=model_args,
                modal="image",
                fsdp=training_args.fsdp,
                do_print=True if local_rank==0 else False
            )
            
            vision_tower = model.get_modal_tower("image")
            vision_tower.to(dtype=torch.bfloat16 if training_args.bf16 else torch.float16, device=training_args.device)
            data_args.image_processor = vision_tower.image_processor
            
            # 관련 config 설정
            model.config.image_aspect_ratio = data_args.image_aspect_ratio
            model.config.image_grid_pinpoints = data_args.image_grid_pinpoints
            
            model.config.mm_use_im_start_end = data_args.mm_use_im_start_end = model_args.mm_use_im_start_end
            training_args.use_im_start_end = model_args.mm_use_im_start_end
            model.config.mm_use_im_patch_token = model_args.mm_use_im_patch_token
            
            
            _init_projection_layer_train(model.get_model().mm_projector,model_args.tune_mm_mlp_adapter,training_args)
            model.initialize_vision_tokenizer(model_args, tokenizer=tokenizer)

        if model_args.load_wo_init_chart==False and model_args.chart_tower is not None :
            model.get_model().initialize_modal_modules(
                model_args=model_args,
                modal="chart",
                fsdp=training_args.fsdp,
                do_print=True if local_rank==0 else False
            )
            
            rank0_print(f"INIT CHART CharTower= {model_args.chart_tower}")
            model.config.mm_use_im_start_end = data_args.mm_use_im_start_end = model_args.mm_use_im_start_end
            
            chart_tower = model.get_modal_tower(modal="chart")
            data_args.chart_processor = chart_tower.chart_processor
            chart_tower.to(dtype=torch.bfloat16 if training_args.bf16 else torch.float16, device=training_args.device)
       
            _init_projection_layer_train(model.get_model().chart_mm_projector,model_args.tune_mm_mlp_adapter,training_args)


    data_module = make_supervised_data_module(tokenizer=tokenizer,
                                              data_args=data_args)
    
  
            
        
    model.generation_config.do_sample=True
    
    trainer = LLaVATrainer(model=model,
                    tokenizer=tokenizer,
                    args=training_args,
                    data_collator=data_module["data_collator"],
                    train_dataset=data_module["train_dataset"] if "train_dataset" in data_module else None,
                    eval_dataset=data_module["eval_dataset"] if "eval_dataset" in data_module else None)
                    # compute_metrics=compute_metrics)
    

    if list(pathlib.Path(training_args.output_dir).glob("checkpoint-*")):
        trainer.train(resume_from_checkpoint=True)
    else:
        trainer.train()
    trainer.save_state()

    model.config.use_cache = True

    if training_args.lora_enable:
        state_dict = get_peft_state_maybe_zero_3(
            model.named_parameters(), training_args.lora_bias
        )
        non_lora_state_dict = get_peft_state_non_lora_maybe_zero_3(
            model.named_parameters()
        )
        if training_args.local_rank == 0 or training_args.local_rank == -1:
            model.config.save_pretrained(training_args.output_dir)
            model.save_pretrained(training_args.output_dir, state_dict=state_dict)
            torch.save(non_lora_state_dict, os.path.join(training_args.output_dir, 'non_lora_trainables.bin'))
    else:
        safe_save_model_for_hf_trainer(trainer=trainer,
                                       output_dir=training_args.output_dir)
    



if __name__ == "__main__":
    train()
