function [points]=precomputeLines(flowLocations, binSize, maxAngle, f)
    % precomputeLines
    %   flowLocations       - Coords of the flow vectors, Struct with fields X: [HxW double], Y: [HxW double]
    %   binSize             - Size of the bins in rad
    %   maxAngle            - Range of rotation to search for in rad. The rotation will be searched within [-maxAngle, maxAngle]
    %   f                   - Focal length in pixels
    %
    % RETURNS
    %   points                - Precomputed lines [H x W x N_POINTS_PER_LINE x 3 double]
    %

    nRotPerDim = maxAngle * 2 / binSize;
    nPointsPerLine = nRotPerDim * 2;

    % Normal vectors to the planes
    eu = cat(3, (flowLocations.X.*flowLocations.Y)./f, -(f+flowLocations.X.^2./f), flowLocations.Y);
    ev = cat(3, f+flowLocations.Y.^2./f, -(flowLocations.X.*flowLocations.Y)./f, -flowLocations.X);

    % Compute the cross product
    vs = cross(eu, ev, 3);
    % Convert to unit vectors
    vs = normalize(vs, 3, 'norm');

    % Points where to sample the line
    x = linspace(0, 1, nPointsPerLine-1);

    % Scale vectors such that z component is -maxAngle
    start_segment=vs./vs(:, :, 3).*-maxAngle;
    % Scale vectors such that z component is +maxAngle
    end_segment=vs./vs(:, :, 3).*maxAngle;

    % Vectorized version of linspace
    % Difference between start and end
    diff = end_segment-start_segment;

    % Outer product for linspace between 0 and diff
    points = reshape(x(:) * diff(:).', [size(x), size(diff)]);
    points = squeeze(points);

    % Add the start point for linespace between start and diff+start (=end)
    start_segment = repmat(start_segment, [1 1 1 nPointsPerLine-1]);
    start_segment = permute(start_segment, [4 1 2 3]);
    points = points + start_segment;

    points = permute(points, [2 3 1 4]);

end
