import numpy as np
from pytracking.evaluation.data import Sequence, BaseDataset, SequenceList
from pytracking.utils.load_text import load_text


class EOTBDataset(BaseDataset):
    """ OTB-2015 dataset

    Publication:
        Object Tracking Benchmark
        Wu, Yi, Jongwoo Lim, and Ming-hsuan Yan
        TPAMI, 2015
        http://faculty.ucmerced.edu/mhyang/papers/pami15_tracking_benchmark.pdf

    Download the dataset from http://cvlab.hanyang.ac.kr/tracker_benchmark/index.html
    """

    def __init__(self):
        super().__init__()
        self.base_path = self.env_settings.eotb_path
        self.sequence_info_list = self._get_sequence_info_list()

    def get_sequence_list(self):
        return SequenceList([self._construct_sequence(s) for s in self.sequence_info_list])

    def _construct_sequence(self, sequence_info):
        sequence_path = sequence_info['path']
        nz = sequence_info['nz']
        ext = sequence_info['ext']
        start_frame = sequence_info['startFrame']
        end_frame = sequence_info['endFrame']

        init_omit = 0
        if 'initOmit' in sequence_info:
            init_omit = sequence_info['initOmit']

        frames = ['{base_path}/{sequence_path}/{frame:0{nz}}.{ext}'.format(base_path=self.base_path,
                                                                           sequence_path=sequence_path, frame=frame_num,
                                                                           nz=nz, ext=ext) for frame_num in
                  range(start_frame + init_omit, end_frame + 1)]

        anno_path = '{}/{}'.format(self.base_path, sequence_info['anno_path'])

        # NOTE: OTB has some weird annos which panda cannot handle
        ground_truth_rect = load_text(str(anno_path), delimiter=(',', None), dtype=np.float64, backend='numpy')

        return Sequence(sequence_info['name'], frames, 'eotb', ground_truth_rect[init_omit:, :],
                        object_class=sequence_info['object_class'])

    def __len__(self):
        return len(self.sequence_info_list)

    def _get_sequence_info_list(self):
        sequence_info_list = [   
    {   'anno_path': 'airplane_mul222/groundtruth_rect.txt',
        'endFrame': 2051,
        'ext': 'jpg',
        'name': 'airplane_mul222',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'airplane_mul222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'bike222/groundtruth_rect.txt',
        'endFrame': 1899,
        'ext': 'jpg',
        'name': 'bike222',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'bike222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'bike333/groundtruth_rect.txt',
        'endFrame': 2001,
        'ext': 'jpg',
        'name': 'bike333',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'bike333/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'bike_low/groundtruth_rect.txt',
        'endFrame': 1290,
        'ext': 'jpg',
        'name': 'bike_low',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'bike_low/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'bottle_mul222/groundtruth_rect.txt',
        'endFrame': 1101,
        'ext': 'jpg',
        'name': 'bottle_mul222',
        'nz': 4,
        'object_class': 'object',
        'path': 'bottle_mul222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'box_hdr/groundtruth_rect.txt',
        'endFrame': 1948,
        'ext': 'jpg',
        'name': 'box_hdr',
        'nz': 4,
        'object_class': 'object',
        'path': 'box_hdr/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'box_low/groundtruth_rect.txt',
        'endFrame': 2084,
        'ext': 'jpg',
        'name': 'box_low',
        'nz': 4,
        'object_class': 'object',
        'path': 'box_low/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'cup222/groundtruth_rect.txt',
        'endFrame': 2010,
        'ext': 'jpg',
        'name': 'cup222',
        'nz': 4,
        'object_class': 'object',
        'path': 'cup222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'cup_low/groundtruth_rect.txt',
        'endFrame': 1933,
        'ext': 'jpg',
        'name': 'cup_low',
        'nz': 4,
        'object_class': 'object',
        'path': 'cup_low/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'cow_mul222/groundtruth_rect.txt',
        'endFrame': 2231,
        'ext': 'jpg',
        'name': 'cow_mul222',
        'nz': 4,
        'object_class': 'animal',
        'path': 'cow_mul222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'dog/groundtruth_rect.txt',
        'endFrame': 642,
        'ext': 'jpg',
        'name': 'dog',
        'nz': 4,
        'object_class': 'animal',
        'path': 'dog/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'fighter_mul/groundtruth_rect.txt',
        'endFrame': 2000,
        'ext': 'jpg',
        'name': 'fighter_mul',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'fighter_mul/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'dog_motion/groundtruth_rect.txt',
        'endFrame': 2788,
        'ext': 'jpg',
        'name': 'dog_motion',
        'nz': 4,
        'object_class': 'animal',
        'path': 'dog_motion/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'dove_motion/groundtruth_rect.txt',
        'endFrame': 2202,
        'ext': 'jpg',
        'name': 'dove_motion',
        'nz': 4,
        'object_class': 'animal',
        'path': 'dove_motion/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'dove_mul222/groundtruth_rect.txt',
        'endFrame': 1297,
        'ext': 'jpg',
        'name': 'dove_mul222',
        'nz': 4,
        'object_class': 'animal',
        'path': 'dove_mul222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'dove_mul/groundtruth_rect.txt',
        'endFrame': 1930,
        'ext': 'jpg',
        'name': 'dove_mul',
        'nz': 4,
        'object_class': 'animal',
        'path': 'dove_mul/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'elephant222/groundtruth_rect.txt',
        'endFrame': 2290,
        'ext': 'jpg',
        'name': 'elephant222',
        'nz': 4,
        'object_class': 'animal',
        'path': 'elephant222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'giraffe222/groundtruth_rect.txt',
        'endFrame': 2390,
        'ext': 'jpg',
        'name': 'giraffe222',
        'nz': 4,
        'object_class': 'animal',
        'path': 'giraffe222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'giraffe_motion/groundtruth_rect.txt',
        'endFrame': 1500,
        'ext': 'jpg',
        'name': 'giraffe_motion',
        'nz': 4,
        'object_class': 'animal',
        'path': 'giraffe_motion/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'giraffe_low/groundtruth_rect.txt',
        'endFrame': 2267,
        'ext': 'jpg',
        'name': 'giraffe_low',
        'nz': 4,
        'object_class': 'animal',
        'path': 'giraffe_low/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'ship_motion/groundtruth_rect.txt',
        'endFrame': 2301,
        'ext': 'jpg',
        'name': 'ship_motion',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'ship_motion/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'ship/groundtruth_rect.txt',
        'endFrame': 967,
        'ext': 'jpg',
        'name': 'ship',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'ship/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'star/groundtruth_rect.txt',
        'endFrame': 1156,
        'ext': 'jpg',
        'name': 'star',
        'nz': 4,
        'object_class': 'object',
        'path': 'star/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'star_motion/groundtruth_rect.txt',
        'endFrame': 2122,
        'ext': 'jpg',
        'name': 'star_motion',
        'nz': 4,
        'object_class': 'object',
        'path': 'star_motion/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'star_mul222/groundtruth_rect.txt',
        'endFrame': 2036,
        'ext': 'jpg',
        'name': 'star_mul222',
        'nz': 4,
        'object_class': 'object',
        'path': 'star_mul222/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'star_mul/groundtruth_rect.txt',
        'endFrame': 2160,
        'ext': 'jpg',
        'name': 'star_mul',
        'nz': 4,
        'object_class': 'object',
        'path': 'star_mul/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'tank_low/groundtruth_rect.txt',
        'endFrame': 2276,
        'ext': 'jpg',
        'name': 'tank_low',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'tank_low/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'tower/groundtruth_rect.txt',
        'endFrame': 1144,
        'ext': 'jpg',
        'name': 'tower',
        'nz': 4,
        'object_class': 'object',
        'path': 'tower/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'tower333/groundtruth_rect.txt',
        'endFrame': 2401,
        'ext': 'jpg',
        'name': 'tower333',
        'nz': 4,
        'object_class': 'object',
        'path': 'tower333/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'truck/groundtruth_rect.txt',
        'endFrame': 1131,
        'ext': 'jpg',
        'name': 'truck',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'truck/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'truck_hdr/groundtruth_rect.txt',
        'endFrame': 1969,
        'ext': 'jpg',
        'name': 'truck_hdr',
        'nz': 4,
        'object_class': 'vehicle',
        'path': 'truck_hdr/inter3_stack',
        'startFrame': 1},
    {   'anno_path': 'whale_mul222/groundtruth_rect.txt',
        'endFrame': 2171,
        'ext': 'jpg',
        'name': 'whale_mul222',
        'nz': 4,
        'object_class': 'animal',
        'path': 'whale_mul222/inter3_stack',
        'startFrame': 1},
        # ----------------------------------------------------------------
    # {   'anno_path': 'cup/groundtruth_rect.txt',
    #     'endFrame': 1115,
    #     'ext': 'jpg',
    #     'name': 'cup',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'cup/img',
    #     'startFrame': 1},
    # {   'anno_path': 'giraffe/groundtruth_rect.txt',
    #     'endFrame': 876,
    #     'ext': 'jpg',
    #     'name': 'giraffe',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'giraffe/img',
    #     'startFrame': 1},
    # {   'anno_path': 'suv_hdr/groundtruth_rect.txt',
    #     'endFrame': 2312,
    #     'ext': 'jpg',
    #     'name': 'suv_hdr',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'suv_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'toy_car/groundtruth_rect.txt',
    #     'endFrame': 1113,
    #     'ext': 'jpg',
    #     'name': 'toy_car',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'toy_car/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cow_mul/groundtruth_rect.txt',
    #     'endFrame': 1900,
    #     'ext': 'jpg',
    #     'name': 'cow_mul',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'cow_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'turtle_motion/groundtruth_rect.txt',
    #     'endFrame': 1700,
    #     'ext': 'jpg',
    #     'name': 'turtle_motion',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'turtle_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ball/groundtruth_rect.txt',
    #     'endFrame': 1104,
    #     'ext': 'jpg',
    #     'name': 'ball',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'ball/img',
    #     'startFrame': 1},
    # {   'anno_path': 'suv_low/groundtruth_rect.txt',
    #     'endFrame': 2171,
    #     'ext': 'jpg',
    #     'name': 'suv_low',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'suv_low/img',
    #     'startFrame': 1},
    # {   'anno_path': 'whale_motion/groundtruth_rect.txt',
    #     'endFrame': 1787,
    #     'ext': 'jpg',
    #     'name': 'whale_motion',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'whale_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'whale/groundtruth_rect.txt',
    #     'endFrame': 928,
    #     'ext': 'jpg',
    #     'name': 'whale',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'whale/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ship_mul222/groundtruth_rect.txt',
    #     'endFrame': 735,
    #     'ext': 'jpg',
    #     'name': 'ship_mul222',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'ship_mul222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bottle_mul111/groundtruth_rect.txt',
    #     'endFrame': 2101,
    #     'ext': 'jpg',
    #     'name': 'bottle_mul111',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'bottle_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'elephant_mul111/groundtruth_rect.txt',
    #     'endFrame': 1751,
    #     'ext': 'jpg',
    #     'name': 'elephant_mul111',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'elephant_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tower_low/groundtruth_rect.txt',
    #     'endFrame': 2209,
    #     'ext': 'jpg',
    #     'name': 'tower_low',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'tower_low/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bear222/groundtruth_rect.txt',
    #     'endFrame': 894,
    #     'ext': 'jpg',
    #     'name': 'bear222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'bear222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'elephant_mul222/groundtruth_rect.txt',
    #     'endFrame': 2066,
    #     'ext': 'jpg',
    #     'name': 'elephant_mul222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'elephant_mul222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'fighter222/groundtruth_rect.txt',
    #     'endFrame': 2070,
    #     'ext': 'jpg',
    #     'name': 'fighter222',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'fighter222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dog_mul111/groundtruth_rect.txt',
    #     'endFrame': 1911,
    #     'ext': 'jpg',
    #     'name': 'dog_mul111',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dog_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dog_mul/groundtruth_rect.txt',
    #     'endFrame': 1840,
    #     'ext': 'jpg',
    #     'name': 'dog_mul',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dog_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'box_motion/groundtruth_rect.txt',
    #     'endFrame': 2150,
    #     'ext': 'jpg',
    #     'name': 'box_motion',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'box_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'fighter/groundtruth_rect.txt',
    #     'endFrame': 952,
    #     'ext': 'jpg',
    #     'name': 'fighter',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'fighter/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dog222/groundtruth_rect.txt',
    #     'endFrame': 1500,
    #     'ext': 'jpg',
    #     'name': 'dog222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dog222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dove_mul111/groundtruth_rect.txt',
    #     'endFrame': 2091,
    #     'ext': 'jpg',
    #     'name': 'dove_mul111',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dove_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'toy_car_low/groundtruth_rect.txt',
    #     'endFrame': 1568,
    #     'ext': 'jpg',
    #     'name': 'toy_car_low',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'toy_car_low/img',
    #     'startFrame': 1},
    # {   'anno_path': 'truck_mul111/groundtruth_rect.txt',
    #     'endFrame': 2001,
    #     'ext': 'jpg',
    #     'name': 'truck_mul111',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'truck_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'star_mul111/groundtruth_rect.txt',
    #     'endFrame': 2351,
    #     'ext': 'jpg',
    #     'name': 'star_mul111',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'star_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ship_mul111/groundtruth_rect.txt',
    #     'endFrame': 2151,
    #     'ext': 'jpg',
    #     'name': 'ship_mul111',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'ship_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'elephant_motion/groundtruth_rect.txt',
    #     'endFrame': 2862,
    #     'ext': 'jpg',
    #     'name': 'elephant_motion',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'elephant_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'whale_mul111/groundtruth_rect.txt',
    #     'endFrame': 2501,
    #     'ext': 'jpg',
    #     'name': 'whale_mul111',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'whale_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ship222/groundtruth_rect.txt',
    #     'endFrame': 685,
    #     'ext': 'jpg',
    #     'name': 'ship222',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'ship222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ship_mul/groundtruth_rect.txt',
    #     'endFrame': 1880,
    #     'ext': 'jpg',
    #     'name': 'ship_mul',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'ship_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'turtle/groundtruth_rect.txt',
    #     'endFrame': 1141,
    #     'ext': 'jpg',
    #     'name': 'turtle',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'turtle/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bear_hdr/groundtruth_rect.txt',
    #     'endFrame': 1845,
    #     'ext': 'jpg',
    #     'name': 'bear_hdr',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'bear_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'box333/groundtruth_rect.txt',
    #     'endFrame': 2569,
    #     'ext': 'jpg',
    #     'name': 'box333',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'box333/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dove/groundtruth_rect.txt',
    #     'endFrame': 1139,
    #     'ext': 'jpg',
    #     'name': 'dove',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dove/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ball_motion/groundtruth_rect.txt',
    #     'endFrame': 1861,
    #     'ext': 'jpg',
    #     'name': 'ball_motion',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'ball_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'turtle222/groundtruth_rect.txt',
    #     'endFrame': 2190,
    #     'ext': 'jpg',
    #     'name': 'turtle222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'turtle222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bear_motion/groundtruth_rect.txt',
    #     'endFrame': 2396,
    #     'ext': 'jpg',
    #     'name': 'bear_motion',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'bear_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'box/groundtruth_rect.txt',
    #     'endFrame': 1134,
    #     'ext': 'jpg',
    #     'name': 'box',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'box/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tank333/groundtruth_rect.txt',
    #     'endFrame': 2601,
    #     'ext': 'jpg',
    #     'name': 'tank333',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'tank333/img',
    #     'startFrame': 1},
    # {   'anno_path': 'elephant/groundtruth_rect.txt',
    #     'endFrame': 956,
    #     'ext': 'jpg',
    #     'name': 'elephant',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'elephant/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ball_hdr/groundtruth_rect.txt',
    #     'endFrame': 2124,
    #     'ext': 'jpg',
    #     'name': 'ball_hdr',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'ball_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'turtle_mul/groundtruth_rect.txt',
    #     'endFrame': 1640,
    #     'ext': 'jpg',
    #     'name': 'turtle_mul',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'turtle_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'airplane_mul/groundtruth_rect.txt',
    #     'endFrame': 1826,
    #     'ext': 'jpg',
    #     'name': 'airplane_mul',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'airplane_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bottle_mul/groundtruth_rect.txt',
    #     'endFrame': 2020,
    #     'ext': 'jpg',
    #     'name': 'bottle_mul',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'bottle_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dove222/groundtruth_rect.txt',
    #     'endFrame': 2800,
    #     'ext': 'jpg',
    #     'name': 'dove222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dove222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'truck_motion/groundtruth_rect.txt',
    #     'endFrame': 2173,
    #     'ext': 'jpg',
    #     'name': 'truck_motion',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'truck_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'toy_car_motion/groundtruth_rect.txt',
    #     'endFrame': 2000,
    #     'ext': 'jpg',
    #     'name': 'toy_car_motion',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'toy_car_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cow_mul111/groundtruth_rect.txt',
    #     'endFrame': 1249,
    #     'ext': 'jpg',
    #     'name': 'cow_mul111',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'cow_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bottle/groundtruth_rect.txt',
    #     'endFrame': 1067,
    #     'ext': 'jpg',
    #     'name': 'bottle',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'bottle/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bike_hdr/groundtruth_rect.txt',
    #     'endFrame': 2645,
    #     'ext': 'jpg',
    #     'name': 'bike_hdr',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'bike_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bottle_motion/groundtruth_rect.txt',
    #     'endFrame': 2050,
    #     'ext': 'jpg',
    #     'name': 'bottle_motion',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'bottle_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'airplane_mul111/groundtruth_rect.txt',
    #     'endFrame': 2391,
    #     'ext': 'jpg',
    #     'name': 'airplane_mul111',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'airplane_mul111/img',
    #     'startFrame': 1},
    # {   'anno_path': 'airplane222/groundtruth_rect.txt',
    #     'endFrame': 1810,
    #     'ext': 'jpg',
    #     'name': 'airplane222',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'airplane222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'turtle_low/groundtruth_rect.txt',
    #     'endFrame': 2091,
    #     'ext': 'jpg',
    #     'name': 'turtle_low',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'turtle_low/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bear/groundtruth_rect.txt',
    #     'endFrame': 893,
    #     'ext': 'jpg',
    #     'name': 'bear',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'bear/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tank_hdr/groundtruth_rect.txt',
    #     'endFrame': 2136,
    #     'ext': 'jpg',
    #     'name': 'tank_hdr',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'tank_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'fighter5/groundtruth_rect.txt',
    #     'endFrame': 1955,
    #     'ext': 'jpg',
    #     'name': 'fighter5',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'fighter5/img',
    #     'startFrame': 1},
    # {   'anno_path': 'dog_mul222/groundtruth_rect.txt',
    #     'endFrame': 2201,
    #     'ext': 'jpg',
    #     'name': 'dog_mul222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'dog_mul222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'airplane/groundtruth_rect.txt',
    #     'endFrame': 1114,
    #     'ext': 'jpg',
    #     'name': 'airplane',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'airplane/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cup_mul/groundtruth_rect.txt',
    #     'endFrame': 2700,
    #     'ext': 'jpg',
    #     'name': 'cup_mul',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'cup_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cup_motion/groundtruth_rect.txt',
    #     'endFrame': 1892,
    #     'ext': 'jpg',
    #     'name': 'cup_motion',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'cup_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'bike_mul/groundtruth_rect.txt',
    #     'endFrame': 2711,
    #     'ext': 'jpg',
    #     'name': 'bike_mul',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'bike_mul/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tower_motion/groundtruth_rect.txt',
    #     'endFrame': 2000,
    #     'ext': 'jpg',
    #     'name': 'tower_motion',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'tower_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cow/groundtruth_rect.txt',
    #     'endFrame': 911,
    #     'ext': 'jpg',
    #     'name': 'cow',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'cow/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tank_motion/groundtruth_rect.txt',
    #     'endFrame': 2002,
    #     'ext': 'jpg',
    #     'name': 'tank_motion',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'tank_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ball_low/groundtruth_rect.txt',
    #     'endFrame': 2121,
    #     'ext': 'jpg',
    #     'name': 'ball_low',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'ball_low/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tank/groundtruth_rect.txt',
    #     'endFrame': 907,
    #     'ext': 'jpg',
    #     'name': 'tank',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'tank/img',
    #     'startFrame': 1},
    # {   'anno_path': 'fighter6/groundtruth_rect.txt',
    #     'endFrame': 1500,
    #     'ext': 'jpg',
    #     'name': 'fighter6',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'fighter6/img',
    #     'startFrame': 1},
    # {   'anno_path': 'tower_hdr/groundtruth_rect.txt',
    #     'endFrame': 2293,
    #     'ext': 'jpg',
    #     'name': 'tower_hdr',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'tower_hdr/img',
    #     'startFrame': 1},
    # {   'anno_path': 'airplane_motion/groundtruth_rect.txt',
    #     'endFrame': 3160,
    #     'ext': 'jpg',
    #     'name': 'airplane_motion',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'airplane_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cow222/groundtruth_rect.txt',
    #     'endFrame': 2450,
    #     'ext': 'jpg',
    #     'name': 'cow222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'cow222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'truck222/groundtruth_rect.txt',
    #     'endFrame': 2200,
    #     'ext': 'jpg',
    #     'name': 'truck222',
    #     'nz': 4,
    #     'object_class': 'vehicle',
    #     'path': 'truck222/img',
    #     'startFrame': 1},
    # {   'anno_path': 'ball333/groundtruth_rect.txt',
    #     'endFrame': 2401,
    #     'ext': 'jpg',
    #     'name': 'ball333',
    #     'nz': 4,
    #     'object_class': 'object',
    #     'path': 'ball333/img',
    #     'startFrame': 1},
    # {   'anno_path': 'cow_motion/groundtruth_rect.txt',
    #     'endFrame': 2422,
    #     'ext': 'jpg',
    #     'name': 'cow_motion',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'cow_motion/img',
    #     'startFrame': 1},
    # {   'anno_path': 'whale222/groundtruth_rect.txt',
    #     'endFrame': 1507,
    #     'ext': 'jpg',
    #     'name': 'whale222',
    #     'nz': 4,
    #     'object_class': 'animal',
    #     'path': 'whale222/img',
    #     'startFrame': 1}
    ]
        return sequence_info_list
