#pragma once

#include <cstdint>

#include <clean-core/allocate.hh>
#include <clean-core/always_false.hh>
#include <clean-core/assert.hh>
#include <clean-core/forward.hh>
#include <clean-core/fwd.hh>

namespace cc
{
/**
 * - single-owner heap-allocated object
 * - move-only type
 * - always uses cc::alloc/cc::free
 * - can only be constructed via make_unique<T>(...)
 * - no polymorphic behavior
 *
 * changes to std::unique_ptr<T>:
 * - no custom deleter
 * - no allocators
 * - no operator<
 * - no operator bool
 * - no T[]
 * - no reset/release
 */
template <class T>
struct unique_ptr
{
    unique_ptr() = default;
    unique_ptr(std::nullptr_t) {}

    unique_ptr(unique_ptr const&) = delete;
    unique_ptr& operator=(unique_ptr const&) = delete;

    unique_ptr(unique_ptr&& rhs) noexcept
    {
        _ptr = rhs._ptr;
        rhs._ptr = nullptr;
    }
    unique_ptr& operator=(unique_ptr&& rhs) noexcept
    {
        static_assert(sizeof(T) > 0, "cannot delete incomplete class");
        // self-move is reset
        if (_ptr)
            cc::free(_ptr);
        _ptr = rhs._ptr;
        rhs._ptr = nullptr;
        return *this;
    }

    ~unique_ptr()
    {
        static_assert(sizeof(T) > 0, "cannot delete incomplete class");
        if (_ptr)
            cc::free(_ptr);
    }

    [[nodiscard]] T* get() const { return _ptr; }

    T* operator->() const
    {
        CC_ASSERT_NOT_NULL(_ptr);
        return _ptr;
    }
    T& operator*() const
    {
        CC_ASSERT_NOT_NULL(_ptr);
        return *_ptr;
    }

    bool operator==(unique_ptr const& rhs) const { return _ptr == rhs._ptr; }
    bool operator!=(unique_ptr const& rhs) const { return _ptr != rhs._ptr; }
    bool operator==(T const* rhs) const { return _ptr == rhs; }
    bool operator!=(T const* rhs) const { return _ptr != rhs; }

    template <typename U, typename... Args>
    friend unique_ptr<U> make_unique(Args&&... args);

    friend bool operator==(T const* lhs, unique_ptr const& rhs) { return lhs == rhs.get(); }
    friend bool operator!=(T const* lhs, unique_ptr const& rhs) { return lhs != rhs.get(); }
    friend bool operator==(std::nullptr_t, unique_ptr const& rhs) { return rhs.get() == nullptr; }
    friend bool operator!=(std::nullptr_t, unique_ptr const& rhs) { return rhs.get() != nullptr; }

private:
    T* _ptr = nullptr;
};

template <class T>
struct unique_ptr<T[]>
{
    static_assert(always_false<T>, "unique_ptr does not support arrays, use cc::vector or cc::array instead");
};

template <typename T, typename... Args>
[[nodiscard]] unique_ptr<T> make_unique(Args&&... args)
{
    unique_ptr<T> p;
    p._ptr = cc::alloc<T>(cc::forward<Args>(args)...);
    return p;
}

template <class T>
struct hash<unique_ptr<T>>
{
    [[nodiscard]] uint64_t operator()(unique_ptr<T> const& v) const noexcept { return uint64_t(v.get()); }
};

template <class T>
struct less<unique_ptr<T>>
{
    [[nodiscard]] bool operator()(unique_ptr<T> const& a, unique_ptr<T> const& b) const noexcept { return a.get() < b.get(); }
};
}
