import torch
import torch.nn as nn
import numpy as np


class PadPrompter(nn.Module):
    def __init__(self, config):
        super(PadPrompter, self).__init__()
        pad_size = config.prompt_size
        image_size = config.data.input_size

        self.base_size = image_size - pad_size*2
        self.pad_up = nn.Parameter(torch.randn([1, config.data.num_segments*3, pad_size, image_size]))
        self.pad_down = nn.Parameter(torch.randn([1, config.data.num_segments*3, pad_size, image_size]))
        self.pad_left = nn.Parameter(torch.randn([1, config.data.num_segments*3, image_size - pad_size*2, pad_size]))
        self.pad_right = nn.Parameter(torch.randn([1, config.data.num_segments*3, image_size - pad_size*2, pad_size]))

    def forward(self, x):
        base = torch.zeros(1, x.shape[1], self.base_size, self.base_size).cuda()
        prompt = torch.cat([self.pad_left, base, self.pad_right], dim=3)
        prompt = torch.cat([self.pad_up, prompt, self.pad_down], dim=2)
        prompt = torch.cat(x.size(0) * [prompt])

        return x + prompt


class FixedPatchPrompter(nn.Module):
    def __init__(self, args):
        super(FixedPatchPrompter, self).__init__()
        self.isize = args.image_size
        self.psize = args.prompt_size
        self.patch = nn.Parameter(torch.randn([1, 3, self.psize, self.psize]))

    def forward(self, x):
        prompt = torch.zeros([1, 3, self.isize, self.isize]).cuda()
        prompt[:, :, :self.psize, :self.psize] = self.patch

        return x + prompt


class RandomPatchPrompter(nn.Module):
    def __init__(self, args):
        super(RandomPatchPrompter, self).__init__()
        self.isize = args.image_size
        self.psize = args.prompt_size
        self.patch = nn.Parameter(torch.randn([1, 3, self.psize, self.psize]))

    def forward(self, x):
        x_ = np.random.choice(self.isize - self.psize)
        y_ = np.random.choice(self.isize - self.psize)

        prompt = torch.zeros([1, 3, self.isize, self.isize]).cuda()
        prompt[:, :, x_:x_ + self.psize, y_:y_ + self.psize] = self.patch

        return x + prompt


def padding(args):
    return PadPrompter(args)


def fixed_patch(args):
    return FixedPatchPrompter(args)


def random_patch(args):
    return RandomPatchPrompter(args)
