import cv2
import numpy as np
from perlin_noise import PerlinNoise
import matplotlib.pyplot as plt
from skimage import img_as_ubyte

# Load the uploaded image

# Define the Perlin Noise blur function for color images
def blur_with_perlin_color(image, magnitude, blur_kernel=(15, 15)):
    """
    Apply Perlin Noise-based blur to a three-channel color image.

    Parameters:
        image (np.ndarray): Input image array.
        octaves (int): Perlin Noise frequency parameter.
        blur_kernel (tuple): Kernel size for Gaussian blur.

    Returns:
        np.ndarray: Image with Perlin Noise-based blur.
        octaves = magnitude
    """
    octaves = magnitude + 1
    xpix, ypix, _ = image.shape

    # Generate Perlin Noise mask
    noise = PerlinNoise(octaves=octaves)
    mask = np.zeros((xpix, ypix), dtype=np.float32)
    for i in range(xpix):
        for j in range(ypix):
            mask[i, j] = noise([i / xpix, j / ypix])

    # Normalize the mask to [0, 1]
    mask = (mask - mask.min()) / (mask.max() - mask.min())

    # Apply Gaussian blur to each channel
    blurred_channels = []
    for c in range(3):  # For each channel (B, G, R)
        channel = image[:, :, c]
        blurred_channel = cv2.GaussianBlur(channel, blur_kernel, 15)
        blended_channel = (1 - mask) * channel + mask * blurred_channel
        blurred_channels.append(blended_channel.astype(np.uint8))

    # Merge channels back together
    output_image = cv2.merge(blurred_channels)
    output_image = img_as_ubyte(output_image)
    return output_image


if __name__ == '__main__':
    image_path = '100014.png'
    image = cv2.imread(image_path)
    image = cv2.resize(image, (1024, 1024))
    octaves_level = 2
    
    perlin = blur_with_perlin_color(image, magnitude = octaves_level)
    
    cv2.imwrite("Perlin_"  + str(octaves_level) + "_" + image_path, perlin)