import matplotlib.pyplot as plt
import numpy as np
from matplotlib.patches import Wedge  # 导入Wedge类

# 简短的标签（方法名）
method_labels = [
    "Depth Map", "Normal Surface", "Albedo Map", 
    "Roughness Map", "Edge Detection", "Binarization", 
    "Illuminance Enhancement", "Image Demoireing", "Image Deblurring", 
    "Absolute Depth", "Brightness", "Contrast", "Saturation", 
    "Box Filter", "Gaussian Blur", "Log Transform", 
    "Gamma Correct", "Gaussian Sharpen", "Bilateral Filter", 
    "Median Filter", "Wavelet Transform", "Perlin Noise", 
    "Dilate", "Erode", "Morphology Opening", "Morphology Closing"
]

LearningBased_labels = [ 
    "Depth Map", "Normal Surface", "Albedo Map", 
    "Roughness Map", "Edge Detection", "Binarization", 
    "Illuminance Enhancement", "Image Demoireing", "Image Deblurring", 
    "Absolute Depth"
]

# 对应的详细描述
descriptions = [
    "Estimate relative distance of images", "Estimate Normal Surface of images", 
    "Estimate Diffuse Albedo of images", "Estimate Roughness of images", 
    "Detect Boundaries of images", "Make images only black and white", 
    "Recover images from dim light", "Recover images from dim light", 
    "Recover images from blurry", "Estimate absolute distance of images", 
    "Increase or Decrease Brightness", "Increase or Decrease Contrast", 
    "Increase or Decrease Saturation", "Smooth images via Box Filter", 
    "Blur images via Gaussian Filter", "Enhance Low Gray part of images", 
    "Adjust exposure of images", "Enhance Edges", "Enhance edges with Blur images", 
    "Enhance edges with Shape images", "Analyze signals in time-frequency domains", 
    "Generate smooth, continuous, random-like patterns", "Enlarge bright areas in images", 
    "Shrink bright areas in binary images", "Erode, then Dilate", "Dilate, then Erode"
]

RuleBased_labels = [
    "Value: (0.1, 1.9]", "Value: (0.1, 1.9]", "Value: (0.1, 1.9]",
    "Kernal: [1, 10]", "Kernal: [1, 10]", "Value: [27, 50]", "Value: [0.5, 1.5]",
    "Value: [0.5, 1.5]", "Sigma: [15, 105]", "Kernal: [1, 19]",
    "LH: [0, 3]", "HH: [4, 6]", "HL: [7, 9]", "Octave level: [0, 9]",
    "Kernal: [0, 9]", "Kernal: [0, 9]", "Kernal: [0, 9]"
]

sizes = [1] * 26  # 每个类别占等分
explode = [0.05] * 10 + [0] * 16  # 让前 10 个部分突出

# 定义颜色：前10个为淡蓝色，后16个为淡黄色
outer_colors = ['lightblue'] * 10 + ['lightyellow'] * 16

# **创建内圈颜色**
inner_colors = [
    "white" if label in LearningBased_labels else "lemonchiffon"  # Learning-Based 镂空，其他为淡黄色
    for label in method_labels
]

# 创建饼图
plt.figure(figsize=(10, 10))

# 画外圈（甜甜圈部分）
wedges, _ = plt.pie(sizes, explode=explode, startangle=90, colors=outer_colors, 
                     wedgeprops={'width': 0.57, 'edgecolor': 'black'})

# **手动绘制内圈**（仅为非 Learning-Based 部分添加黑色边框）
ax = plt.gca()

for i, label in enumerate(method_labels):
    angle = (i / 26) * 360 + 90  # 计算每个扇区的中心角度
    wedge = None
    if label in LearningBased_labels:
        # 对于镂空部分，不绘制边框
        wedge = Wedge(center=(0, 0), r=0.43, theta1=angle, theta2=angle + (360 / 26), 
                      facecolor=inner_colors[i], edgecolor='none', lw=0)
    else:
        # 对于非镂空部分，保留边框
        wedge = Wedge(center=(0, 0), r=0.43, theta1=angle, theta2=angle + (360 / 26), 
                      facecolor=inner_colors[i], edgecolor='black', lw=1)

    ax.add_patch(wedge)

# 设置图表标题
plt.title("Distribution of 26 Image Processing Techniques", fontsize=14)

# **新的左右半圆逻辑**
left_half_labels = method_labels[:13]   # "Depth Map" 到 "Saturation"
left_half_descs = descriptions[:13]

right_half_labels = method_labels[13:]  # "Box Filter" 到 "Morphology Closing"
right_half_descs = descriptions[13:]

# 遍历所有 wedges，调整 label & description
for i, (wedge, label, description) in enumerate(zip(wedges, method_labels, descriptions)):
    angle = (wedge.theta1 + wedge.theta2) / 2  # 获取当前扇形的中心角度

    # 计算文本位置
    if label in LearningBased_labels:
        x = 0.5 * np.cos(np.radians(angle))  # label 稍微内侧
        y = 0.5 * np.sin(np.radians(angle))

        x_desc = 1.1 * np.cos(np.radians(angle))  # description 更外侧
        y_desc = 1.1 * np.sin(np.radians(angle))
        
    else:
        x = 0.45 * np.cos(np.radians(angle))  # label 稍微内侧
        y = 0.45 * np.sin(np.radians(angle))

        x_desc = 1.05 * np.cos(np.radians(angle))  # description 更外侧
        y_desc = 1.05 * np.sin(np.radians(angle))

    # **判断是否在“右半圆”（即 Box Filter ~ Morphology Closing）**
    if label in right_half_labels:
        label_rotation = angle  # 右半圆的 label 旋转 180°，正向显示
        desc_rotation = angle
        ha = "left"  # 靠近中心
    else:
        label_rotation = angle + 180  # 左半圆的 label 依旧旋转 180°
        desc_rotation = angle + 180
        ha = "right"

    # 添加标签 (method_labels)，字体较大
    plt.text(x, y, label, ha=ha, va='center', fontsize=8, 
             rotation=label_rotation, rotation_mode='anchor', fontweight='bold')

    # 添加描述 (descriptions)，字体较小
    #plt.text(x_desc, y_desc, description, ha=ha, va='center', fontsize=8, rotation=desc_rotation, rotation_mode='anchor')

# 保存图表为图片
plt.savefig("pie_chart_without_description.png", bbox_inches='tight', dpi=300)
plt.savefig("pie_chart_without_description.pdf", bbox_inches='tight', dpi=300)

# 关闭图表，释放内存
plt.close()
