import io
import logging
import shutil
import sys
import tempfile
import unittest
from pathlib import Path

from drone_base.config.logger import LoggerSetup


class TestLoggerSetup(unittest.TestCase):
    def setUp(self):
        self.test_dir = Path(tempfile.mkdtemp())
        self.stdout = io.StringIO()
        self.original_stdout = sys.stdout
        sys.stdout = self.stdout

    def tearDown(self):
        shutil.rmtree(self.test_dir)
        sys.stdout = self.original_stdout
        logging.getLogger().handlers = []

    def test_console_logger_creation(self):
        """Test that console logger is created with correct settings"""
        logger = LoggerSetup.setup_logger("test_console")

        self.assertEqual(len(logger.handlers), 1)
        handler = logger.handlers[0]

        self.assertIsInstance(handler, logging.StreamHandler)
        self.assertEqual(handler.level, logging.INFO)

        logger.info("Test message")
        output = self.stdout.getvalue()

        self.assertRegex(output, r'\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d{3}')  # timestamp
        self.assertIn('[INFO    ]', output)  # level name with padding
        self.assertIn('test_console', output)  # logger name
        self.assertIn('Test message', output)  # message

    def test_file_logger_creation(self):
        """Test that file logger is created with correct settings"""
        log_file = self.test_dir / "test.log"
        logger = LoggerSetup.setup_logger("test_file", log_file=log_file)

        self.assertEqual(len(logger.handlers), 2)
        file_handler = next(h for h in logger.handlers if isinstance(h, logging.handlers.RotatingFileHandler))

        self.assertEqual(file_handler.level, logging.DEBUG)
        self.assertEqual(file_handler.maxBytes, 5 * 1024 * 1024)
        self.assertEqual(file_handler.backupCount, 5)

        test_message = "Test file message"
        logger.debug(test_message)

        with open(log_file, 'r') as f:
            content = f.read()

        self.assertIn('[DEBUG   ]', content)
        self.assertIn('test_file', content)
        self.assertIn(test_message, content)
        self.assertIn('Thread:', content)
        self.assertIn('PID:', content)

    def test_different_log_levels(self):
        """Test that different log levels work correctly"""
        logger = LoggerSetup.setup_logger(
            "test_levels",
            level=logging.DEBUG,
            console_level=logging.WARNING
        )

        logger.info("Info message")
        self.assertEqual(self.stdout.getvalue(), "")

        logger.warning("Warning message")
        self.assertIn("Warning message", self.stdout.getvalue())

    def test_log_directory_creation(self):
        """Test that log directory is created if it doesn't exist"""
        log_file = self.test_dir / "subdir" / "test.log"
        LoggerSetup.setup_logger("test_dir_creation", log_file=log_file)

        self.assertTrue(log_file.parent.exists())
        self.assertTrue(log_file.exists())

    def test_level_name_padding(self):
        """Test that level names are padded correctly"""
        logger = LoggerSetup.setup_logger("test_padding")

        for level, name in [
            (logging.DEBUG, 'DEBUG'),
            (logging.INFO, 'INFO'),
            (logging.WARNING, 'WARNING'),
            (logging.ERROR, 'ERROR'),
            (logging.CRITICAL, 'CRITICAL')
        ]:
            self.stdout.seek(0)
            self.stdout.truncate()

            logger.log(level, "Test message")
            output = self.stdout.getvalue()

            if level >= logging.INFO:
                padded_level = f'[{name:<{LoggerSetup.LEVEL_WIDTH}}]'
                self.assertIn(padded_level, output)


if __name__ == '__main__':
    unittest.main()
