import queue
import unittest
from unittest.mock import Mock, patch, MagicMock

import numpy as np

from drone_base.config.logger import LoggerSetup
from drone_base.config.video import VideoConfig
from drone_base.control.drone_commander import DroneCommander
from drone_base.stream.display_only_processor import DisplayOnlyProcessor


class TestDisplayOnlyProcessor(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.video_config = Mock(spec=VideoConfig)
        self.video_config.height = 720
        self.video_config.width = 1280
        self.drone_commander = Mock(spec=DroneCommander)
        self.frame_queue = queue.Queue()
        self.test_frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        self.test_frame[300:400, 500:700, 0] = 255
        self.mock_logger = MagicMock()
        patcher = patch.object(LoggerSetup, 'setup_logger', return_value=self.mock_logger)
        self.addCleanup(patcher.stop)
        patcher.start()

        self.processor = DisplayOnlyProcessor(
            video_config=self.video_config,
            drone_commander=self.drone_commander,
            frame_queue=self.frame_queue,
        )

    def test_initialization(self):
        """Test proper initialization of the display only processor."""
        self.assertEqual(self.processor.config, self.video_config)
        self.assertEqual(self.processor.frame_queue, self.frame_queue)
        self.assertEqual(self.processor.drone_commander, self.drone_commander)
        self.assertFalse(self.processor.is_frame_saved)
        self.assertFalse(hasattr(self.processor, 'frame_saver'))

    def test_process_frame(self):
        """Test that the process_frame method returns the frame unchanged."""
        processed_frame = self.processor._process_frame(self.test_frame)

        self.assertTrue(np.array_equal(processed_frame, self.test_frame))

        original_test_frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        original_test_frame[300:400, 500:700, 0] = 255
        self.assertTrue(np.array_equal(self.test_frame, original_test_frame))

    @patch("threading.Thread")
    def test_start_display_processor(self, mock_thread):
        """Test starting the display processor thread."""
        self.processor.start()

        self.assertTrue(self.processor._running.is_set())
        mock_thread.assert_called_once()

        _, thread_kwargs = mock_thread.call_args
        self.assertTrue(thread_kwargs["daemon"])
        self.assertEqual(thread_kwargs["name"], "DisplayOnlyProcessorThread")

    @patch("cv2.imshow")
    @patch("cv2.waitKey")
    def test_display_functionality(self, mock_wait_key, mock_imshow):
        """Test that frames are correctly passed to display."""
        self.frame_queue.put(self.test_frame)

        _ = self.processor._run_processing_loop

        def run_once():
            self.processor._running.set()
            try:
                frame = self.frame_queue.get(timeout=0.1)
                with self.processor._lock:
                    self.processor._display_frame(self.processor._process_frame(frame))
            except queue.Empty:
                pass
            self.processor._running.clear()

        with patch.object(self.processor, '_run_processing_loop', side_effect=run_once):
            self.processor.start()
            import time
            time.sleep(0.2)

        mock_imshow.assert_called_once_with("Parrot Olympe Video Stream", self.test_frame)
        mock_wait_key.assert_called_once_with(1)


if __name__ == "__main__":
    unittest.main()
