from pathlib import Path

import torch

from nn.saving import save_model


class EarlyStopping:
    def __init__(self, patience: int = 7, verbose: bool = False, delta: float = 0, save_dir: str | Path | None = None):
        """
        :param patience: How long to wait after last time validation loss improved.
        :param verbose: If True, prints a message for each validation loss improvement.
        :param delta: Minimum change in the monitored quantity to qualify as an improvement.
        """
        self.patience = patience
        self.verbose = verbose
        self.delta = delta
        self.save_dir = save_dir
        if self.save_dir is not None:
            self.save_dir = Path(save_dir)
            self.save_dir.mkdir(parents=True, exist_ok=True)

        self.counter = 0
        self.best_score = None
        self.early_stop = False
        self.val_loss_min = torch.inf
        self.best_model = None

    def __call__(self, val_loss, model, model_name: str | None = None):
        score = -val_loss

        if self.best_score is None:
            self.best_score = score
            self.save_checkpoint(val_loss, model, model_name)
        elif score < self.best_score + self.delta:
            self.counter += 1
            if self.verbose:
                print(f'EarlyStopping counter: {self.counter} out of {self.patience}')
            if self.counter >= self.patience:
                self.early_stop = True
        else:
            self.best_score = score
            self.save_checkpoint(val_loss, model, model_name)
            self.counter = 0

    def save_checkpoint(self, val_loss, model, model_name: str | None = None):
        """Saves model when validation loss decreases."""
        if self.verbose:
            print(f'Validation loss decreased ({self.val_loss_min:.6f} --> {val_loss:.6f}).  Saving model ...')
        self.val_loss_min = val_loss
        self.best_model = model.state_dict()
        if self.save_dir is not None and model_name is not None:
            save_model(model=model, save_dir=self.save_dir, model_name=model_name)
