from dataclasses import dataclass, field
from enum import Enum


class DroneIp(str, Enum):
    SIMULATED = "10.202.0.1"
    CABLE = "192.168.53.1"
    WIRELESS = "192.168.42.1"

    def __str__(self) -> str:
        return self.value


class GimbalType(str, Enum):
    REF_ABSOLUTE = "absolute"
    REF_RELATIVE = "relative"
    MODE_POSITION = "position"
    MODE_VELOCITY = "velocity"

    def __str__(self) -> str:
        return self.value


@dataclass(frozen=True)
class DroneSpeed:
    """
    Configuration class for drone piloting commands.

    All values for x, y, z and z_rot must be positive and less than or equal to 100, where:
        - x (roll): represents translation left (negative) and right (positive) on X axis .
        - y (pitch): represents translation forwards (positive) and backwards (negative) on Y axis.
        - z (yaw):  represents translation upwards (positive) and backwards (negative) on Z axis.
        - z_rot (gaz): represents rotation around Z axis (rotation on Yaw), left (negative) and right (positive).

    The sign of the values for x, y, and z and z_rot are the responsibility of the user of this class.

    Time step dt (piloting_time) is in seconds and must be a positive float.
    """
    x: int = field(default=20, metadata={"range": (0, 100)})
    y: int = field(default=20, metadata={"range": (0, 100)})
    z: int = field(default=20, metadata={"range": (0, 100)})
    z_rot: int = field(default=20, metadata={"range": (0, 100)})
    dt: float = field(default=0.1)

    def __post_init__(self):
        dt_value = getattr(self, "dt")
        if dt_value < 0:
            raise ValueError(f"Time step must be positive, got \"{dt_value}\" instead")
        for attr in ("x", "y", "z", "z_rot"):
            value = getattr(self, attr)
            if not (0 <= value <= 100):  # noqa: PLR2004
                raise ValueError(f"{attr} must be between [0, 100], got \"{value}\" instead")
