import random
from dataclasses import dataclass
from enum import IntEnum

import math

from drone_base.config.drone import GimbalType


@dataclass
class TiltCommand:
    pitch_deg: float
    control_mode: GimbalType = GimbalType.MODE_POSITION
    reference_type: GimbalType = GimbalType.REF_ABSOLUTE


@dataclass
class MovementByCommand:
    forward: float = 0
    right: float = 0
    down: float = 0
    rotation: float = 0

    is_random_movement: bool = False
    will_also_land_command: bool = False


@dataclass
class PilotingCommand:
    x: int = 0
    y: int = 0
    z: int = 0
    rotation: int = 0
    dt: float = 0.15

    is_random_movement: bool = False
    will_also_land_command: bool = False


@dataclass
class CommandSequence:
    commands: list[MovementByCommand | TiltCommand]


# TODO: extend this for more possible movements, maybe use the same approach as in manual/operations.py
class Movement(IntEnum):
    FORWARD = 1
    TURN_LEFT = 2
    TURN_RIGHT = 3
    U_TURN = 4
    RANDOM = 5


class MovementCommandFactory:
    def __init__(self, forward_speed: float = 0.3):
        self.forward_speed = forward_speed
        self.rotation_speed = math.pi / 3
        self.pi_over_two = math.pi / 2
        self.commands = {
            Movement.FORWARD: MovementByCommand(forward=self.forward_speed),
            Movement.TURN_LEFT: MovementByCommand(rotation=-self.rotation_speed),
            Movement.TURN_RIGHT: MovementByCommand(rotation=self.rotation_speed),
            Movement.U_TURN: MovementByCommand(rotation=self.pi_over_two),
        }
        self._possible_turns = [self.rotation_speed, -self.rotation_speed]

    def get_command(self, movement: Movement) -> MovementByCommand | None:
        """If the movement is random then it will create a new instance since they are dynamic by convention."""
        if movement == Movement.RANDOM:
            rotation = random.choice(self._possible_turns)
            return MovementByCommand(rotation=rotation, is_random_movement=True)
        return self.commands[movement]


class PilotingCommandFactory:
    def __init__(self, forward_speed: int = 35, rotation_speed: int = 35):
        self.forward_speed = forward_speed
        self.rotation_speed = rotation_speed
        self.rotation_dt = 0.5
        self.commands = {
            Movement.FORWARD: PilotingCommand(y=self.forward_speed),
            Movement.TURN_LEFT: PilotingCommand(rotation=-self.rotation_speed, dt=self.rotation_dt),
            Movement.TURN_RIGHT: PilotingCommand(rotation=self.rotation_speed, dt=self.rotation_dt),
            Movement.U_TURN: PilotingCommand(rotation=self.rotation_speed, dt=self.rotation_dt),
        }
        self._possible_turns = [self.rotation_speed, -self.rotation_speed]

    def get_piloting_command(self, movement: Movement) -> PilotingCommand | None:
        if movement == Movement.RANDOM:
            rotation = random.choice(self._possible_turns)
            return PilotingCommand(rotation=rotation, dt=self.rotation_dt, is_random_movement=True)
        return self.commands[movement]
