import numpy as np
from math import radians, cos, sin, sqrt, atan2
from scipy.spatial.transform import Rotation


def quat_to_rotation_matrix(quat: dict[str, float]) -> Rotation:
    """Convert quaternion to rotation matrix"""
    return Rotation.from_quat([quat['x'], quat['y'], quat['z'], quat['w']])


def rotate_velocity(quat: dict[str, float], velocity: dict[str, float]) -> np.ndarray:
    """Rotate velocity vector using quaternion"""
    rotation = quat_to_rotation_matrix(quat)
    vel_vector = np.array([velocity['north'], velocity['east'], velocity['down']])
    return rotation.apply(vel_vector)


def quaternion_to_euler(quat: dict[str, float]) -> np.ndarray:
    """Convert quaternion to euler angles (roll, pitch, yaw)."""
    w, x, y, z = quat['w'], quat['x'], quat['y'], quat['z']

    # Roll (x-axis rotation)
    sinr_cosp = 2 * (w * x + y * z)
    cosr_cosp = 1 - 2 * (x * x + y * y)
    roll = np.arctan2(sinr_cosp, cosr_cosp)

    # Pitch (y-axis rotation)
    sinp = 2 * (w * y - z * x)
    pitch = np.arcsin(sinp)

    # Yaw (z-axis rotation)
    siny_cosp = 2 * (w * z + x * y)
    cosy_cosp = 1 - 2 * (y * y + z * z)
    yaw = np.arctan2(siny_cosp, cosy_cosp)

    return np.array([roll, pitch, yaw])


def get_forward_vector(quat: dict[str, float]) -> np.ndarray:
    """Get the forward direction in drone's local frame vector from quaternion."""
    euler = quaternion_to_euler(quat)
    yaw = euler[2]

    forward = np.array([
        np.cos(yaw),  # x component
        np.sin(yaw),  # y component
        0  # z component (assuming level flight)
    ])
    return forward


def haversine_distance(lat1: float, lon1: float, lat2: float, lon2: float) -> float:
    """Calculate the distance between two GPS points in meters."""
    earth_radius_in_meters = 6371000

    lat1, lon1, lat2, lon2 = map(radians, [lat1, lon1, lat2, lon2])

    dlat = lat2 - lat1
    dlon = lon2 - lon1

    a = sin(dlat / 2) ** 2 + cos(lat1) * cos(lat2) * sin(dlon / 2) ** 2
    c = 2 * atan2(sqrt(a), sqrt(1 - a))

    return earth_radius_in_meters * c
