import unittest

from drone_base.config.video import VideoConfig


class TestVideoConfig(unittest.TestCase):
    def test_default_initialization(self):
        config = VideoConfig()
        self.assertEqual(config.width, 640)
        self.assertEqual(config.height, 360)
        self.assertEqual(config.frame_center_x, 320)
        self.assertEqual(config.frame_center_y, 180)
        self.assertIsNone(config.cam_mode)

    def test_custom_initialization(self):
        config = VideoConfig(width=1280, height=720)
        self.assertEqual(config.width, 1280)
        self.assertEqual(config.height, 720)
        self.assertEqual(config.frame_center_x, 640)
        self.assertEqual(config.frame_center_y, 360)
        self.assertIsNone(config.cam_mode)

    def test_invalid_width_initialization(self):
        with self.assertRaises(ValueError) as context:
            VideoConfig(width=0, height=360)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

        with self.assertRaises(ValueError) as context:
            VideoConfig(width=-100, height=360)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

    def test_invalid_height_initialization(self):
        with self.assertRaises(ValueError) as context:
            VideoConfig(width=640, height=0)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

        with self.assertRaises(ValueError) as context:
            VideoConfig(width=640, height=-100)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

    def test_aspect_ratio(self):
        config = VideoConfig(width=1920, height=1080)
        self.assertAlmostEqual(config.aspect_ratio(), 1.78, places=2)  # 16:9 aspect ratio

        config = VideoConfig(width=640, height=480)
        self.assertAlmostEqual(config.aspect_ratio(), 1.33, places=2)  # 4:3 aspect ratio

    def test_update_dimensions(self):
        config = VideoConfig()
        config.update_dimensions(1920, 1080)
        self.assertEqual(config.width, 1920)
        self.assertEqual(config.height, 1080)
        self.assertEqual(config.frame_center_x, 960)
        self.assertEqual(config.frame_center_y, 540)

    def test_invalid_update_dimensions(self):
        config = VideoConfig()

        with self.assertRaises(ValueError) as context:
            config.update_dimensions(0, 1080)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

        with self.assertRaises(ValueError) as context:
            config.update_dimensions(1920, -1080)
        self.assertEqual(str(context.exception), "Video width and height must be greater than zero.")

        self.assertEqual(config.width, 640)
        self.assertEqual(config.height, 360)

    def test_frame_center_computation(self):
        """Test that frame center is correctly computed for various dimensions"""
        test_cases = [
            (800, 600, 400, 300),
            (1920, 1080, 960, 540),
            (1280, 720, 640, 360),
            (640, 480, 320, 240)
        ]

        for width, height, expected_center_x, expected_center_y in test_cases:
            with self.subTest(width=width, height=height):
                config = VideoConfig(width=width, height=height)
                self.assertEqual(config.frame_center_x, expected_center_x)
                self.assertEqual(config.frame_center_y, expected_center_y)

    def test_valid_cam_mode(self):
        """Test initialization with valid camera modes"""
        config_photo = VideoConfig(cam_mode="photo")
        self.assertEqual(config_photo.cam_mode, "photo")

        config_recording = VideoConfig(cam_mode="recording")
        self.assertEqual(config_recording.cam_mode, "recording")

    def test_invalid_cam_mode(self):
        """Test initialization with invalid camera modes"""
        invalid_modes = ["video", "preview", "1080p", "", 123]

        for invalid_mode in invalid_modes:
            with self.subTest(mode=invalid_mode):
                with self.assertRaises(ValueError) as context:
                    VideoConfig(cam_mode=invalid_mode)
                expected_error = (
                    f"Invalid camera mode for: \"{invalid_mode}\". "
                    f"Possible values are \"photo\", \"recording\" or default None."
                )
                self.assertEqual(str(context.exception), expected_error)

    def test_cam_mode_with_dimensions(self):
        """Test that camera mode works correctly with custom dimensions"""
        config = VideoConfig(width=1024, height=768, cam_mode="photo")
        self.assertEqual(config.width, 1024)
        self.assertEqual(config.height, 768)
        self.assertEqual(config.cam_mode, "photo")
        self.assertEqual(config.frame_center_x, 512)
        self.assertEqual(config.frame_center_y, 384)
        self.assertEqual(config.frame_area, 786432)


if __name__ == "__main__":
    unittest.main()
