import unittest
from unittest.mock import MagicMock

from pynput.keyboard import KeyCode

from drone_base.config.drone import DroneSpeed
from drone_base.control.manual.keyboard_controller import KeyboardController


class TestKeyboardController(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.mock_drone_commander = MagicMock()
        self.speed_config = DroneSpeed()
        self.controller = KeyboardController(self.mock_drone_commander, self.speed_config)

    def tearDown(self):
        """Clean up after each test method."""
        self.controller.stop()

    def test_initialization(self):
        """Test initial state of KeyboardController."""
        self.assertIsNone(self.controller.listener)
        self.assertEqual(self.controller.movement_vector.x, 0)
        self.assertEqual(self.controller.movement_vector.y, 0)
        self.assertEqual(self.controller.movement_vector.z, 0)
        self.assertEqual(self.controller.movement_vector.z_rot, 0)

    def test_start_stop(self):
        """Test starting and stopping the keyboard listener."""
        self.controller.start()
        self.assertIsNotNone(self.controller.listener)
        self.assertTrue(self.controller.listener.running)

        self.controller.stop()
        self.assertIsNone(self.controller.listener)

    def test_movement_keys(self):
        """Test handling of movement key presses and releases."""
        self.controller._on_press_button(KeyCode.from_char('i'))
        self.assertEqual(self.controller.movement_vector.y, self.speed_config.y)
        self.mock_drone_commander.piloting.assert_called_with(
            x=0,
            y=self.speed_config.y,
            z_rot=0,
            z=0,
            dt=self.speed_config.dt
        )

        self.controller._on_release_button(KeyCode.from_char('i'))
        self.assertEqual(self.controller.movement_vector.y, 0)

    def test_multiple_movement_keys(self):
        """Test handling multiple movement keys pressed simultaneously."""
        self.controller._on_press_button(KeyCode.from_char('i'))
        self.controller._on_press_button(KeyCode.from_char('l'))

        self.assertEqual(self.controller.movement_vector.y, self.speed_config.y)
        self.assertEqual(self.controller.movement_vector.x, -self.speed_config.x)

        self.controller._on_release_button(KeyCode.from_char('i'))
        self.assertEqual(self.controller.movement_vector.y, 0)
        self.assertEqual(self.controller.movement_vector.x, -self.speed_config.x)

    def test_invalid_key_press(self):
        """Test handling of invalid key presses."""
        invalid_key = MagicMock()
        del invalid_key.char

        try:
            self.controller._on_press_button(invalid_key)
            self.controller._on_release_button(invalid_key)
        except AttributeError:
            self.fail("Controller should handle invalid keys gracefully")

    def test_command_keys_takeoff(self):
        """Test takeoff command key."""
        self.controller._on_press_button(KeyCode.from_char('t'))
        self.mock_drone_commander.take_off.assert_called_once()

    def test_command_keys_land(self):
        """Test land command key."""
        self.controller._on_press_button(KeyCode.from_char('y'))
        self.mock_drone_commander.land.assert_called_once()

    def test_command_keys_photo(self):
        """Test photo command key."""
        self.controller._on_press_button(KeyCode.from_char('p'))
        # Can't assert logger calls directly, but we can verify no commander method was called
        self.mock_drone_commander.take_off.assert_not_called()
        self.mock_drone_commander.land.assert_not_called()
        self.mock_drone_commander.tilt_camera.assert_not_called()

    def test_command_keys_gimbal(self):
        """Test gimbal control command keys."""
        from drone_base.config.drone import GimbalType
        angle_keys = {
            "z": 0,
            "x": -15,
            "c": -30,
            "v": -45,
            "s": -60,
            "d": -75,
            "f": -90,
        }

        for key, angle in angle_keys.items():
            self.controller._on_press_button(KeyCode.from_char(key))
            self.mock_drone_commander.tilt_camera.assert_called_with(
                pitch_deg=angle,
                reference_type=GimbalType.REF_ABSOLUTE
            )

            self.mock_drone_commander.reset_mock()

    def test_emergency_stop(self):
        """Test emergency stop functionality."""
        self.controller._on_press_button(KeyCode.from_char('i'))
        self.assertEqual(self.controller.movement_vector.y, self.speed_config.y)

        self.controller._on_press_button(KeyCode.from_char('q'))
        self.assertFalse(self.controller.is_running)
        self.assertIsNone(self.controller.listener)

    def test_all_movement_keys(self):
        """Test all movement keys for complete coverage."""
        # Test forward/backward (i/k)
        self.controller._on_press_button(KeyCode.from_char('i'))
        self.assertEqual(self.controller.movement_vector.y, self.speed_config.y)
        self.controller._on_release_button(KeyCode.from_char('i'))

        self.controller._on_press_button(KeyCode.from_char('k'))
        self.assertEqual(self.controller.movement_vector.y, -self.speed_config.y)
        self.controller._on_release_button(KeyCode.from_char('k'))

        # Test left/right (j/l)
        self.controller._on_press_button(KeyCode.from_char('j'))
        self.assertEqual(self.controller.movement_vector.x, self.speed_config.x)
        self.controller._on_release_button(KeyCode.from_char('j'))

        self.controller._on_press_button(KeyCode.from_char('l'))
        self.assertEqual(self.controller.movement_vector.x, -self.speed_config.x)
        self.controller._on_release_button(KeyCode.from_char('l'))

        # Test rotate left/right (u/o)
        self.controller._on_press_button(KeyCode.from_char('o'))
        self.assertEqual(self.controller.movement_vector.z_rot, self.speed_config.z_rot)
        self.controller._on_release_button(KeyCode.from_char('o'))

        self.controller._on_press_button(KeyCode.from_char('u'))
        self.assertEqual(self.controller.movement_vector.z_rot, -self.speed_config.z_rot)
        self.controller._on_release_button(KeyCode.from_char('u'))

        # Test up/down (n/m)
        self.controller._on_press_button(KeyCode.from_char('n'))
        self.assertEqual(self.controller.movement_vector.z, self.speed_config.z)
        self.controller._on_release_button(KeyCode.from_char('n'))

        self.controller._on_press_button(KeyCode.from_char('m'))
        self.assertEqual(self.controller.movement_vector.z, -self.speed_config.z)
        self.controller._on_release_button(KeyCode.from_char('m'))

    def test_logger_setup(self):
        """Test logger setup with a specific directory."""
        import tempfile
        from pathlib import Path

        with tempfile.TemporaryDirectory() as temp_dir:
            temp_path = Path(temp_dir)
            controller = KeyboardController(
                self.mock_drone_commander,
                self.speed_config,
                logger_dir=temp_path
            )
            expected_log_file = temp_path / "KeyboardController.log"
            self.assertIsNotNone(controller.logger)
            controller.stop()
            self.assertTrue(expected_log_file.exists())

    def test_command_exception_handling(self):
        """Test handling of exceptions during command execution."""
        self.mock_drone_commander.take_off.side_effect = Exception("Test exception")

        try:
            self.controller._on_press_button(KeyCode.from_char('t'))
        except Exception:
            self.fail("Exception was not properly handled")

        self.mock_drone_commander.take_off.assert_called_once()

    def test_non_moving_state(self):
        """Test that piloting is not called when the movement vector is all zeros."""
        self.mock_drone_commander.reset_mock()

        self.controller._on_press_button(KeyCode.from_char('i'))
        self.controller._on_release_button(KeyCode.from_char('i'))
        self.mock_drone_commander.reset_mock()
        self.controller._update_movement()

        self.mock_drone_commander.piloting.assert_not_called()


if __name__ == '__main__':
    unittest.main()
