import shutil
import tempfile
import unittest
from pathlib import Path
from unittest.mock import Mock, patch

import cv2
import numpy as np

from drone_base.stream.saving.frames_to_video import CreateVideoFromFrames, create_video


class TestCreateVideoFromFrames(unittest.TestCase):
    def setUp(self):
        self.test_dir = tempfile.mkdtemp()
        self.input_dir = Path(self.test_dir) / "frames"
        self.input_dir.mkdir(parents=True)

        self.frame_timestamps = [1000, 1033, 1066, 1100]  # 30fps approximately
        self.create_test_frames()

    def tearDown(self):
        shutil.rmtree(self.test_dir)

    def create_test_frames(self):
        """Helper method to create test frame files"""
        test_frame = np.zeros((100, 100, 3), dtype=np.uint8)
        for i, timestamp in enumerate(self.frame_timestamps):
            frame_path = self.input_dir / f"frame_{i:06d}_{timestamp}.png"
            cv2.imwrite(str(frame_path), test_frame)

    def test_init(self):
        """Test initialization of CreateVideoFromFrames"""
        creator = CreateVideoFromFrames(self.input_dir)
        self.assertEqual(str(creator.input_dir), str(self.input_dir))
        self.assertEqual(creator.output_path, "output.mp4")
        self.assertEqual(creator.frame_type, "png")

    def test_populate_frame_paths(self):
        """Test frame path population and sorting"""
        creator = CreateVideoFromFrames(self.input_dir)
        creator._CreateVideoFromFrames__populate_frame_paths()

        self.assertEqual(len(creator.frame_files), len(self.frame_timestamps))
        self.assertEqual(creator.timestamps, self.frame_timestamps)

    def test_populate_frame_paths_empty_directory(self):
        """Test handling of empty input directory"""
        empty_dir = Path(self.test_dir) / "empty"
        empty_dir.mkdir()

        creator = CreateVideoFromFrames(empty_dir)
        with self.assertRaises(FileNotFoundError):
            creator._CreateVideoFromFrames__populate_frame_paths()

    def test_compute_fps(self):
        """Test FPS calculation"""
        creator = CreateVideoFromFrames(self.input_dir)
        creator._CreateVideoFromFrames__populate_frame_paths()

        fps, duration = creator._CreateVideoFromFrames__compute_fps()

        # Expected duration: 100ms (1100 - 1000)
        self.assertAlmostEqual(duration, 100.0, places=1)
        # Expected FPS: 4 frames / 0.1 seconds = 30 fps
        self.assertAlmostEqual(fps, 40.0, places=1)

    def test_analyze_frame_timing(self):
        """Test frame timing analysis"""
        creator = CreateVideoFromFrames(self.input_dir)
        creator._CreateVideoFromFrames__populate_frame_paths()

        timing_stats = creator._CreateVideoFromFrames__analyze_frame_timing()

        self.assertIn("avg_fps", timing_stats)
        self.assertIn("min_fps", timing_stats)
        self.assertIn("max_fps", timing_stats)
        self.assertIn("std_dev", timing_stats)

    @patch("cv2.VideoWriter")
    def test_create_video(self, mock_video_writer):
        """Test video creation process"""
        mock_writer = Mock()
        mock_video_writer.return_value = mock_writer

        output_path = str(Path(self.test_dir) / "output.mp4")
        creator = CreateVideoFromFrames(self.input_dir, output_path)

        creator.create_video()

        mock_video_writer.assert_called_once()
        self.assertEqual(mock_writer.write.call_count, len(self.frame_timestamps))
        mock_writer.release.assert_called_once()

    @patch("cv2.VideoWriter")
    def test_create_video_with_missing_frames(self, mock_video_writer):
        """Test video creation with some unreadable frames"""
        mock_writer = Mock()
        mock_video_writer.return_value = mock_writer

        invalid_frame_path = self.input_dir / "frame_999999_2000.png"
        invalid_frame_path.touch()

        creator = CreateVideoFromFrames(self.input_dir)
        creator.create_video()

        self.assertEqual(mock_writer.write.call_count, len(self.frame_timestamps))

    def test_create_video_function(self):
        """Test the convenience function create_video"""
        with patch("drone_base.stream.saving.frames_to_video.CreateVideoFromFrames") as mock_creator_class:
            mock_instance = Mock()
            mock_creator_class.return_value = mock_instance

            create_video("input_dir", "output.mp4", "png")

            mock_creator_class.assert_called_once_with(
                input_dir="input_dir",
                output_path="output.mp4",
                frame_type="png"
            )
            mock_instance.create_video.assert_called_once()


if __name__ == "__main__":
    unittest.main()
