import queue
import shutil
import tempfile
import threading
import time
import unittest
from unittest.mock import Mock, patch, MagicMock

import cv2
import numpy as np

from drone_base.config.logger import LoggerSetup
from drone_base.config.video import VideoConfig
from drone_base.control.drone_commander import DroneCommander
from drone_base.stream.base_video_processor import BaseVideoProcessor
from drone_base.stream.saving.frame_saver import BufferedFrameSaver


class ConcreteVideoProcessor(BaseVideoProcessor):
    def _process_frame(self, frame: np.ndarray) -> np.ndarray:
        return frame


class TestBaseVideoProcessor(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.video_config = Mock(spec=VideoConfig)
        self.video_config.height = 720
        self.video_config.width = 1280
        self.drone_commander = Mock(spec=DroneCommander)
        self.frame_queue = queue.Queue()
        self.log_dir = tempfile.mkdtemp()
        self.frame_save_dir = tempfile.mkdtemp()
        self.test_frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        self.mock_logger = MagicMock()
        patcher = patch.object(LoggerSetup, 'setup_logger', return_value=self.mock_logger)
        self.addCleanup(patcher.stop)
        patcher.start()

        self.processor = ConcreteVideoProcessor(
            video_config=self.video_config,
            drone_commander=self.drone_commander,
            frame_queue=self.frame_queue,
            logger_dir=self.log_dir,
            frame_save_dir=self.frame_save_dir
        )

    def tearDown(self):
        """Clean up after each test."""
        self.processor.stop()
        while not self.frame_queue.empty():
            try:
                self.frame_queue.get_nowait()
            except queue.Empty:
                break
        self.processor._running.clear()
        time.sleep(0.1)
        cv2.destroyAllWindows()

        shutil.rmtree(self.log_dir, ignore_errors=True)
        shutil.rmtree(self.frame_save_dir, ignore_errors=True)

    def test_initialization(self):
        """Test proper initialization of the processor."""
        processor = ConcreteVideoProcessor(
            video_config=self.video_config,
            drone_commander=self.drone_commander,
            frame_queue=self.frame_queue,
            logger_dir=self.log_dir,
            frame_save_dir=self.frame_save_dir
        )

        self.assertEqual(processor.config, self.video_config)
        self.assertEqual(processor.frame_queue, self.frame_queue)
        self.assertEqual(processor.drone_commander, self.drone_commander)
        self.assertTrue(processor.is_frame_saved)
        self.assertIsInstance(processor.frame_saver, BufferedFrameSaver)

        processor_no_save = ConcreteVideoProcessor(
            video_config=self.video_config,
            drone_commander=self.drone_commander,
            frame_queue=self.frame_queue,
            logger_dir=self.log_dir,
            frame_save_dir=None
        )

        self.assertFalse(processor_no_save.is_frame_saved)
        self.assertFalse(hasattr(processor_no_save, 'frame_saver'))

        processor_no_logger = ConcreteVideoProcessor(
            video_config=self.video_config,
            drone_commander=self.drone_commander,
            frame_queue=self.frame_queue,
            logger_dir=None,
            frame_save_dir=None
        )

        self.assertTrue(hasattr(processor_no_logger, 'logger'))

    @patch("threading.Thread")
    def test_start(self, mock_thread):
        """Test starting the frame processor."""
        self.processor.start()

        self.assertTrue(self.processor._running.is_set())
        mock_thread.assert_called_once()

        _, thread_kwargs = mock_thread.call_args
        self.assertTrue(thread_kwargs["daemon"])
        self.assertEqual(thread_kwargs["name"], "ConcreteVideoProcessorThread")

        self.mock_logger.info.assert_called_with("%s Frame processing thread started.", "ConcreteVideoProcessor")

    def test_stop(self):
        """Test stopping the frame processor."""
        self.processor._running.set()
        self.processor.stop()

        self.assertFalse(self.processor._running.is_set())

    def test_stop_with_frame_saving(self):
        """Test stopping when frame saving is enabled."""
        self.assertTrue(self.processor.is_frame_saved)

        with patch.object(self.processor.frame_saver, "save_all") as mock_save:
            self.processor.stop()
            mock_save.assert_called_once()

    @patch("cv2.destroyAllWindows")
    def test_frame_display_context(self, mock_destroy):
        """Test the frame display context manager."""
        with self.processor._frame_display_context():
            pass
        mock_destroy.assert_called_once()

    @patch("cv2.destroyAllWindows")
    def test_frame_display_context_with_exception(self, mock_destroy):
        """Test the frame display context manager handles exceptions."""
        with self.assertRaises(ValueError):
            with self.processor._frame_display_context():
                raise ValueError("Test exception")
        mock_destroy.assert_called_once()

    def test_process_frame_implementation(self):
        """Test that the concrete implementation of _process_frame works."""
        frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        processed_frame = self.processor._process_frame(frame)

        self.assertTrue(np.array_equal(processed_frame, frame))

    @patch("cv2.imshow")
    @patch("cv2.waitKey")
    def test_display_frame(self, mock_wait_key, mock_imshow):
        """Test frame display functionality."""
        frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        self.processor._display_frame(frame)

        mock_imshow.assert_called_once_with("Parrot Olympe Video Stream", frame)
        mock_wait_key.assert_called_once_with(1)

    def test_run_processing_loop_frame_saving(self):
        """Test frame saving in the processing loop."""
        test_frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        self.frame_queue.put(test_frame)

        with patch.object(self.processor, "_display_frame"):
            with patch.object(self.processor, "_process_frame", return_value=test_frame):
                with patch.object(self.processor.frame_saver, "add_frame") as mock_add_frame:
                    self.processor._running.set()
                    thread = threading.Thread(target=self.processor._run_processing_loop)
                    thread.start()
                    time.sleep(0.2)
                    self.processor.stop()
                    thread.join(timeout=1.0)

                    mock_add_frame.assert_called_once()
                    args, kwargs = mock_add_frame.call_args
                    self.assertTrue(np.array_equal(kwargs["frame"], test_frame))
                    self.assertIsInstance(kwargs["timestamp"], float)

    def test_run_processing_loop_empty_queue(self):
        """Test processing loop behavior with empty queue."""
        with patch.object(self.processor, "_display_frame") as mock_display:
            self.processor._running.set()
            thread = threading.Thread(target=self.processor._run_processing_loop)
            thread.start()
            time.sleep(0.2)
            self.processor.stop()
            thread.join(timeout=1.0)

            mock_display.assert_not_called()

    def test_run_processing_loop_exception_handling(self):
        """Test exception handling in processing loop."""
        test_frame = np.zeros((720, 1280, 3), dtype=np.uint8)
        self.frame_queue.put(test_frame)

        with patch.object(self.processor, "_process_frame", side_effect=Exception("Test error")):
            self.processor._running.set()
            thread = threading.Thread(target=self.processor._run_processing_loop)
            thread.start()
            time.sleep(0.2)
            self.processor.stop()
            thread.join(timeout=1.0)

            self.mock_logger.error.assert_called_with("Unable to process frame...")
            self.mock_logger.critical.assert_called_once()


if __name__ == "__main__":
    unittest.main()
