class VideoConfig:
    """Configuration class for the drone video stream with validation"""

    def __init__(self, width: int = 640, height: int = 360, cam_mode: str | None = None, save_extension: str = "png"):
        """
        Sets the width and height of the video stream. Computes the center of the frame.

        :param width: Width of the video stream in pixels, by default 640 (half of the PDRAW stream)
        :param height: Height of the video stream in pixels, by default 360 (half of the PDRAW stream)
        :param cam_mode: Camera mode to use, either "photo" or "recording" or default None.
                          Photo mode has a stream for 1024x768. Recording mode is the default and it's 1280x720.
        """
        if width <= 0 or height <= 0:
            raise ValueError("Video width and height must be greater than zero.")
        if save_extension not in ("png", "jpg", "jpeg"):
            raise ValueError(f"Save extension must be 'png' or 'jpg' or 'jpeg'. Provided {save_extension}")

        self.width: int = width
        self.height: int = height
        self.frame_center_x, self.frame_center_y = self.__compute_frame_center()
        self.frame_area = self.__compute_frame_area()
        self.max_queue_size = 30
        self.save_extension = save_extension

        if cam_mode not in ("photo", "recording", None):
            raise ValueError(
                f"Invalid camera mode for: \"{cam_mode}\". "
                f"Possible values are \"photo\", \"recording\" or default None."
            )

        self.cam_mode = cam_mode

    def __compute_frame_center(self) -> tuple[int, int]:
        return self.width // 2, self.height // 2

    def __compute_frame_area(self) -> int:
        return self.width * self.height

    def aspect_ratio(self) -> float:
        """Compute the aspect ratio of the video frame."""
        return self.width / self.height

    def update_dimensions(self, width: int, height: int):
        if width <= 0 or height <= 0:
            raise ValueError("Video width and height must be greater than zero.")
        self.width = width
        self.height = height
        self.frame_center_x, self.frame_center_y = self.__compute_frame_center()
        self.frame_area = self.__compute_frame_area()
