import contextlib
import random
import threading
from pathlib import Path
from time import sleep, time

import olympe
from olympe.enums.ardrone3.PilotingState import FlyingStateChanged_State
from olympe.messages import gimbal
from olympe.messages.ardrone3.Piloting import moveBy, Landing, TakeOff
from olympe.messages.ardrone3.PilotingState import FlyingStateChanged
from olympe.messages.ardrone3.SpeedSettings import MaxRotationSpeed
from olympe.messages.ardrone3.SpeedSettingsState import MaxRotationSpeedChanged

from drone_base.config.drone import GimbalType
from drone_base.config.logger import LoggerSetup
from drone_base.control.operations import MovementByCommand, MovementCommandFactory, TiltCommand, CommandSequence, \
    PilotingCommandFactory, PilotingCommand
from drone_base.control.states import DroneState, FlightState, GimbalOrientation


class DroneCommander:
    """Handles drone flight control and command processing"""

    # TODO: remove unnecessary code such as forward moves, will the movement be random, action history etc...
    #  Discuss with peers about what to keep and what to not keep and what else to add.
    def __init__(self, drone: olympe.Drone, logger_dir: str | Path | None = None):
        self.drone = drone
        self.state = DroneState()
        if logger_dir is not None:
            logger_dir = Path(logger_dir) / f"{self.__class__.__name__}.log"
        self.logger = LoggerSetup.setup_logger(logger_name=self.__class__.__name__, log_file=logger_dir)

        self._command_thread = None
        self._command_semaphore = threading.Semaphore(1)
        self.forward_moves = 0
        self.random_moves = random.randint(5, 10)
        self.command_factory = MovementCommandFactory(forward_speed=0.3)
        self.piloting_command_factory = PilotingCommandFactory(forward_speed=35, rotation_speed=100)
        self.__SECONDS_TO_TIMEOUT = 10.0
        self.__ANGLE_TOLERANCE = 1.0
        self.action_history = []
        self.distance_forward = 0

        self.is_piloting_search = True
        if self.is_piloting_search:
            self.piloting_history = []

    # TODO: maybe add sleeping for important commands, same as __wait_for_tilt()
    def connect(self, retries: int = 3) -> bool:
        self.logger.info("Connecting to drone...")
        return self.drone.connect(retry=retries)

    def disconnect(self) -> bool:
        self.logger.info("Disconnecting from drone...")
        if self.state.get_state()[0] == FlightState.FLYING:
            self.land()
        return self.drone.disconnect()

    def take_off(self):
        """Execute drone takeoff"""
        self.logger.info("Taking off...")
        result = self.drone(TakeOff()).wait().success()
        if result:
            self.state.set_flight_state(FlightState.FLYING)
        return result

    def land(self):
        """Execute drone landing"""
        self.logger.info("Landing...")
        result = self.drone(Landing()).wait().success()
        if result:
            self.state.set_flight_state(FlightState.LANDED)
        return result

    def move_by(self, forward: float, right: float, down: float, rotation: float):
        """Execute relative movement"""
        result = self.drone(
            moveBy(forward, right, down, rotation)
            >> FlyingStateChanged(state="hovering", _timeout=3)
        ).wait()
        return result.success()

    def piloting(self, x: int, y: int, z: int, z_rot: int, dt: float):
        """Execute direct piloting commands."""
        return self.drone.piloting(x, y, z_rot, z, dt)

    def tilt_camera(
            self,
            pitch_deg: float,
            control_mode: GimbalType = GimbalType.MODE_POSITION,
            reference_type: GimbalType = GimbalType.REF_ABSOLUTE
    ):
        """
        Tilts the gimbal camera for the pitch value.

        :param pitch_deg: The pitch value in degrees.
                          Positive value will rotate the camera upwards.
                          Negative value will rotate the camera downwards.
        :param control_mode: "position" or "velocity" control mode.
        :param reference_type: An absolute reference type considers the camera to be parallel to the ground plane.
                               A relative reference type takes into consideration the current position of the camera.
                               This parameter influences the pitch_deg value.
        """
        curr_state = self.state.gimbal_orientation
        self.state.set_gimbal_orientation(GimbalOrientation.IN_BETWEEN)
        camera_action = self.drone(gimbal.set_target(
            gimbal_id=0,
            control_mode=control_mode,
            yaw_frame_of_reference="none",
            yaw=0,
            pitch_frame_of_reference=reference_type,
            pitch=pitch_deg,
            roll_frame_of_reference="none",
            roll=0,
        )).wait()
        if not camera_action.success():
            self.logger.warning("Cannot tilt gimbal camera.")
            self.state.set_gimbal_orientation(curr_state)
            return

        self.__wait_for_tilt(pitch_deg=pitch_deg)
        self.state.set_gimbal_orientation(self.state.get_orientation_on_pitch(pitch_value=pitch_deg))

    def __wait_for_tilt(self, pitch_deg: float):
        start_time = time()
        while time() - start_time < self.__SECONDS_TO_TIMEOUT:
            gimbal_state = self.drone.get_state(gimbal.attitude)
            current_pitch = gimbal_state[0]['pitch_absolute']

            self.logger.info("Current pitch: %s", current_pitch)
            if abs(current_pitch - pitch_deg) <= self.__ANGLE_TOLERANCE:
                self.logger.info("Reached pitch tolerance. Changing gimbal state.")
                break

            sleep(0.1)

        if time() - start_time >= self.__SECONDS_TO_TIMEOUT:
            self.logger.warning("Timeout waiting for gimbal to reach target position")

    def change_rotation_speed(self, speed: int) -> None:
        self.logger.info("Changing rotation speed... Current speed: %s", self.drone.get_state(MaxRotationSpeedChanged))
        self.drone(MaxRotationSpeed(current=speed).wait())
        sleep(1)
        self.logger.info("New rotation speed: %s", self.drone.get_state(MaxRotationSpeedChanged))

    def has_crashed(self):
        """Check if the drone is in an emergency state"""
        flying_state = self.drone.get_state(FlyingStateChanged)["state"]

        if flying_state == FlyingStateChanged_State.emergency:
            self.logger.warning("Drone is in an emergency state!!!")
            return True
        elif flying_state == FlyingStateChanged_State.landed:
            self.logger.warning("Drone is in a landed state. Might be a crash.")
            return True

        return False

    def execute_command(
            self, command: MovementByCommand | PilotingCommand | TiltCommand | CommandSequence, is_blocking: bool = True
    ):
        self._command_thread = threading.Thread(
            target=self._send_command,
            args=(command, is_blocking)
        )
        self._command_thread.start()

    def _send_command(
            self, command: MovementByCommand | PilotingCommand | TiltCommand | CommandSequence, is_blocking: bool = True
    ):
        if not self._command_semaphore.acquire(blocking=is_blocking):
            return "Cannot send the command."

        with contextlib.ExitStack() as stack:
            stack.callback(self._command_semaphore.release)
            self.logger.info({"action": "send_command", "command": vars(command)})
            self.action_history.append(command)

            if isinstance(command, PilotingCommand):
                self.__handle_piloting_command(command)
            if isinstance(command, MovementByCommand):
                self.__handle_movement_by_command(command)
            elif isinstance(command, TiltCommand):
                self.__handle_tilt_command(command)
            elif isinstance(command, CommandSequence):
                for cmd in command.commands:
                    if isinstance(cmd, MovementByCommand):
                        self.__handle_movement_by_command(cmd)
                    elif isinstance(cmd, TiltCommand):
                        self.__handle_tilt_command(cmd)

            return "OK"

    def __handle_tilt_command(self, command: TiltCommand):
        self.tilt_camera(
            pitch_deg=command.pitch_deg,
            control_mode=command.control_mode,
            reference_type=command.reference_type
        )

    def __handle_movement_by_command(self, command: MovementByCommand):
        if command.forward > 0:
            self.forward_moves += 1
            self.distance_forward += abs(command.forward)
        if command.is_random_movement:
            self.forward_moves = 0
            self.random_moves = random.randint(5, 10)
            self.logger.info("Random movement: %s", vars(command))

        self.move_by(forward=command.forward, right=command.right, down=command.down, rotation=command.rotation)
        if command.will_also_land_command:
            self.land()

    def __handle_piloting_command(self, command: PilotingCommand):
        if command.y > 0:
            self.forward_moves += 1
            self.distance_forward += abs(command.y)
        if command.is_random_movement:
            self.forward_moves = 0
            self.random_moves = random.randint(5, 10)
            self.logger.info("Random movement: %s", vars(command))

        success = self.piloting(x=command.x, y=command.y, z=command.z, z_rot=command.rotation, dt=command.dt)
        if success:
            sleep(command.dt)
            self.piloting_history.append(command)
        if command.will_also_land_command:
            self.land()
