from enum import Enum
from threading import Lock


class GimbalOrientation(Enum):
    """Represents the possible gimbal camera orientations."""
    HORIZONTAL = 0
    TILTED = -30
    DOWNWARD = -90

    """This is for inbetween states"""
    IN_BETWEEN = 9999


class FlightState(str, Enum):
    INITIAL = "initial-landed"
    LANDED = "landed"
    FLYING = "flying"

    def __str__(self) -> str:
        return self.value


class DroneState:
    """Tracks the current state of the drone."""

    def __init__(self):
        self.flight_state = FlightState.INITIAL
        self.gimbal_orientation = GimbalOrientation.HORIZONTAL
        self._lock = Lock()
        self.__max_pitch_val = 90

    def set_flight_state(self, state: FlightState):
        with self._lock:
            self.flight_state = state

    def set_gimbal_orientation(self, orientation: GimbalOrientation):
        with self._lock:
            self.gimbal_orientation = orientation

    def get_orientation_on_pitch(self, pitch_value: float) -> GimbalOrientation:
        """
        With pitch value (in degrees) computes the orientation of the drone.
        Uses the absolute value since only magnitude affects orientation.
        """
        pitch_value = abs(pitch_value)
        if pitch_value == 0:
            return GimbalOrientation.HORIZONTAL
        elif 0 < pitch_value < self.__max_pitch_val:
            return GimbalOrientation.TILTED
        return GimbalOrientation.DOWNWARD

    def get_state(self) -> tuple[FlightState, GimbalOrientation]:
        return self.flight_state, self.gimbal_orientation
