import os
import random
from abc import ABC
from pathlib import Path
from time import sleep, time

import olympe

from drone_base.config.drone import DroneIp, DroneSpeed
from drone_base.config.video import VideoConfig
from drone_base.control.drone_commander import DroneCommander
from drone_base.control.manual.keyboard_controller import KeyboardController
from drone_base.control.operations import TiltCommand
from drone_base.control.states import FlightState
from drone_base.stream.base_video_processor import BaseVideoProcessor
from drone_base.stream.stream_handler import StreamHandler
from drone_base.utils.readable_time import date_time_now_to_file_name

olympe.log.update_config({"loggers": {"olympe": {"level": "WARNING"}}})


class BaseStreamingController(ABC):
    """Base class for all streaming controllers with common functionality."""

    def __init__( # noqa: PLR0913
            self,
            ip: DroneIp,
            processor_class: type[BaseVideoProcessor],
            speed: int = 35,
            log_path: str | Path | None = None,
            results_path: str | Path | None = None,
            video_config: VideoConfig | None = None,
    ):
        self.start_time = date_time_now_to_file_name()
        if log_path is not None:
            log_path = Path(log_path) / self.start_time
        if results_path is not None:
            results_path = Path(results_path) / self.start_time

        self.video_config = video_config if video_config is not None else VideoConfig()

        self.drone = olympe.Drone(ip)
        self.drone_commander = DroneCommander(self.drone, logger_dir=log_path)
        self.streaming_manager = StreamHandler(
            self.drone, self.video_config, logger_dir=log_path, metadata_dir=results_path
        )
        self.frame_processor = processor_class(
            video_config=self.video_config, frame_queue=self.streaming_manager.frame_queue,
            drone_commander=self.drone_commander, logger_dir=log_path, frame_save_dir=results_path
        )
        self.manual_controller = KeyboardController(
            commander=self.drone_commander,
            speed_config=DroneSpeed(x=speed, y=speed, z=speed, z_rot=100, dt=0.15)
        )

        # TODO: create maybe a json config file for this an other params (video config, speeds).
        self.will_stop_after_mission = False
        self.is_random_start = True
        self.is_piloting_search = True
        self.started_mission_time = None
        self.timeout_mission_seconds = 30

    def start(self):
        """Starts all components of the streaming controller."""
        if not self.drone_commander.connect():
            raise RuntimeError("Failed to connect to the drone...")
        self.streaming_manager.start_streaming()
        self.manual_controller.start()
        self.frame_processor.start()

        if self.is_piloting_search:
            self.drone_commander.change_rotation_speed(speed=100)

        if self.will_stop_after_mission:
            self.drone_commander.take_off()
            sleep(10)
            if self.is_random_start:
                start_right_displacement = random.choice([-0.2, 0.2])
                start_down_displacement = random.choice([0, 0.1, 0.2, 0.3])
                print(f"Moving right: {start_right_displacement} m | Moving down: {start_down_displacement} m ")
                if not self.drone_commander.move_by(
                        forward=0, right=start_right_displacement, down=start_down_displacement, rotation=0):
                    self.drone_commander.move_by(
                        forward=0, right=start_right_displacement, down=start_down_displacement, rotation=0)
                sleep(5)
            self.drone_commander.execute_command(TiltCommand(pitch_deg=-30), is_blocking=False)
            if self.started_mission_time is None:
                self.started_mission_time = time()

    def stop(self):
        """Stop all components and clean up resources."""
        self.manual_controller.stop()
        self.streaming_manager.stop_streaming()
        self.frame_processor.stop()
        self.drone_commander.disconnect()

    def run(self):
        try:
            self.start()
            while self.should_continue_running():
                sleep(1)
        finally:
            self.stop()

    def should_continue_running(self) -> bool:
        """Determine if the controller should continue running after landing."""
        if not self.manual_controller.is_running:
            return False

        if self.will_stop_after_mission:
            if self.drone_commander.state.flight_state == FlightState.LANDED:
                sleep(3)
                return False
            time_passed = int(time() - self.started_mission_time)
            if time_passed % 2 == 0 and time_passed > self.timeout_mission_seconds:
                if self.drone_commander.has_crashed():
                    print("Mission timed out. Emergency. Exiting abnormally.")
                    os._exit(1)
        return True
